<?php

/**
 * Class MtboricaBoricaHelper
 *
 * Helper class for BORICA payment gateway configuration constants and utility methods.
 * Contains public keys, URLs, transaction types, and other BORICA-related constants.
 */
class MtboricaBoricaHelper
{
    /**
     * BORICA test environment public key for signature verification.
     */
    public const BORICA_TEST_PUBLIC_KEY =
    '-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAya0nWBwWR19j/B8STchu
oADV295eP0nd0I3KWIeiiiPV4+xfzqOVguKOt086BrIRLAfTU46dURtwX3PaqiJw
fXa8lpr1kQWCqQH6q/nl6t9A5OOBWF34pFvxgRL64QaQgUTwP+l4sx4p6JFKV41y
itFrgnWaz9X/Y6SXGDTFKcRfDy1FrRTY6g+UTAJtPTUOA8yi53kSK2lO8P3+Bzr1
paBVLjvsSt+uj4Jbz1ssY2IeHqaZm3vW4he6A20Z/ZGE/n1+YQoEqP4NIXVAjrlJ
W+/Z5hvokGWEdf6Fmyz+gA3G+pgVIbiTovW2SgPBy0H6runURtYS6oM3FhPRGJ2Q
uQIDAQAB
-----END PUBLIC KEY-----';

    /**
     * BORICA production environment public key for signature verification.
     */
    public const BORICA_PRODUCTION_PUBLIC_KEY =
    '-----BEGIN PUBLIC KEY-----
MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEA8oqRwrBQKZdO+VPoDHFf
5giPRQkObyvXM8wDDm+kIPhC4gIR8Ch9sFZlQxa8ZE3cCDMsAviub6+RvTtkqy1p
C5abVJQhAIpmIX3NDf82+aD+kGuxIe6JpcFAfKhV0zEr5LzqDYNzhn2huDpv7W+Z
5zUjtwxP5Ob9/Lmw0ckF6XE3drzt0pK26p3ZKRicUh/cGBWQC7bGHpnSnNmvF5Fq
b6PLu6Gzq5RjtSnJG7q8T7DWL5iFVpSFMN0tLbfuCM0ZSc5xodrk84esRm36KMV+
lx3t6HQ1kvs7aQKbGq0TtBAbfQRlYBlgV2DamyOQfH6vMiD179bol4Ss0XvaYWzq
fwIDAQAB
-----END PUBLIC KEY-----';

    /**
     * Get the appropriate public key based on test mode setting.
     *
     * @param bool $testMode True for test mode, false for production
     * @return string Public key string
     */
    public static function getPublicKey($testMode = true)
    {
        return $testMode ? self::BORICA_TEST_PUBLIC_KEY : self::BORICA_PRODUCTION_PUBLIC_KEY;
    }

    /**
     * Get the appropriate public key based on test mode configuration.
     *
     * @return string Public key string
     */
    public static function getPublicKeyFromConfig()
    {
        $testMode = (int) Configuration::get('mtborica_testmode');
        return self::getPublicKey($testMode === 1);
    }

    /**
     * Signs authorization data for BORICA payment gateway.
     *
     * @param string $borica_terminal Terminal ID
     * @param int $borica_trtype Transaction type
     * @param string $borica_amount Amount
     * @param string $borica_currency Currency code (BGN or EUR)
     * @param string $borica_order Order number
     * @param string $borica_timestamp Timestamp
     * @param string $borica_nonce Nonce
     * @return array Array with 'boricaError' and 'pSign' keys
     */
    public static function signAuthorization(
        $borica_terminal,
        $borica_trtype,
        $borica_amount,
        $borica_currency,
        $borica_order,
        $borica_timestamp,
        $borica_nonce
    ) {
        $result = [];
        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        $borica_test_key = '';
        $borica_production_key = '';
        $borica_test_password = '';
        $borica_production_password = '';

        if ('BGN' == $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_bgn');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_bgn');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_bgn');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_bgn');
        }
        if ('EUR' == $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_eur');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_eur');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_eur');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_eur');
        }
        if (1 == $borica_testmode) {
            $priv_key = $borica_test_key;
            $priv_key_password = $borica_test_password;
        } else {
            $priv_key = $borica_production_key;
            $priv_key_password = $borica_production_password;
        }
        $data =
            strlen($borica_terminal) . $borica_terminal .
            strlen($borica_trtype) . $borica_trtype .
            strlen($borica_amount) . $borica_amount .
            strlen($borica_currency) . $borica_currency .
            strlen($borica_order) . $borica_order .
            strlen($borica_timestamp) . $borica_timestamp .
            strlen($borica_nonce) . $borica_nonce .
            '-';

        try {
            $pkeyid = openssl_get_privatekey($priv_key, $priv_key_password);
            openssl_sign($data, $signature, $pkeyid, OPENSSL_ALGO_SHA256);
            $result['boricaError'] = '';
            $result['pSign'] = strtoupper(bin2hex($signature));
        } catch (\Exception $e) {
            // Log detailed error for debugging (server-side only)
            if (class_exists('MtboricaLogger')) {
                MtboricaLogger::error('BORICA signAuthorization error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
            }
            // Return generic error message (no sensitive information exposed)
            $result['boricaError'] = 'Signature generation failed';
            $result['pSign'] = '';
        }

        return $result;
    }

    /**
     * Signs payment status check request data for BORICA payment gateway.
     *
     * This method generates a digital signature (P_SIGN) for checking payment status
     * by signing the concatenated transaction parameters using the appropriate private key
     * based on currency and test/production mode.
     *
     * @param string $borica_terminal Terminal ID from BORICA configuration
     * @param string|int $borica_trtype Transaction type (typically for status check)
     * @param string $borica_order Order number/identifier
     * @param string $borica_nonce Unique nonce value for the transaction
     * @param string $borica_currency Currency code (BGN or EUR) - determines which key to use
     * @return array Array containing:
     *               - 'boricaError' (string): Empty string on success, error message on failure
     *               - 'pSign' (string): Hexadecimal signature in uppercase, empty string on error
     */
    public static function signCheckPayment(
        $borica_terminal,
        $borica_trtype,
        $borica_order,
        $borica_nonce,
        $borica_currency
    ) {
        $result = [];
        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        if ('BGN' === $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_bgn');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_bgn');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_bgn');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_bgn');
        }
        if ('EUR' === $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_eur');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_eur');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_eur');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_eur');
        }
        if (1 === $borica_testmode) {
            $priv_key = $borica_test_key;
            $priv_key_password = $borica_test_password;
        } else {
            $priv_key = $borica_production_key;
            $priv_key_password = $borica_production_password;
        }
        $data =
            strlen($borica_terminal) . $borica_terminal .
            strlen($borica_trtype) . $borica_trtype .
            strlen($borica_order) . $borica_order .
            strlen($borica_nonce) . $borica_nonce;
        try {
            $pkeyid = openssl_get_privatekey($priv_key, $priv_key_password);
            openssl_sign($data, $signature, $pkeyid, OPENSSL_ALGO_SHA256);
            $result['boricaError'] = '';
            $result['pSign'] = strtoupper(bin2hex($signature));
        } catch (\Exception $e) {
            // Log detailed error for debugging (server-side only)
            if (class_exists('MtboricaLogger')) {
                MtboricaLogger::error('BORICA signCheckPayment error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
            }
            // Return generic error message (no sensitive information exposed)
            $result['boricaError'] = 'Signature generation failed';
            $result['pSign'] = '';
        }
        return $result;
    }

    /**
     * Verifies the digital signature of a BORICA payment gateway response.
     *
     * This method validates that the response from BORICA has not been tampered with
     * by verifying the P_SIGN signature against the concatenated response parameters
     * using the appropriate public key based on test/production mode.
     *
     * @param string $borica_p_sign Hexadecimal signature from BORICA response (P_SIGN)
     * @param string $borica_action Action code from BORICA response (empty string if not provided)
     * @param string $borica_rc Response code from BORICA (empty string if not provided)
     * @param string $borica_approval Approval code from BORICA (empty string if not provided)
     * @param string $borica_terminal Terminal ID from BORICA response (empty string if not provided)
     * @param string|int $borica_trtype Transaction type (empty string if not provided)
     * @param string|float $borica_amount Transaction amount (empty string if not provided)
     * @param string $borica_currency Currency code (empty string if not provided)
     * @param string $borica_order Order number/identifier (empty string if not provided)
     * @param string $borica_rrn Retrieval Reference Number (empty string if not provided)
     * @param string $borica_int_ref Internal reference (empty string if not provided)
     * @param string $borica_pares_status PARES status from 3D Secure (empty string if not provided)
     * @param string $borica_eci Electronic Commerce Indicator (empty string if not provided)
     * @param string $borica_timestamp Transaction timestamp (empty string if not provided)
     * @param string $borica_nonce Unique nonce value (empty string if not provided)
     * @return bool True if signature is valid, false if invalid or verification error occurs
     */
    public static function checkAuthorization(
        $borica_p_sign,
        $borica_action,
        $borica_rc,
        $borica_approval,
        $borica_terminal,
        $borica_trtype,
        $borica_amount,
        $borica_currency,
        $borica_order,
        $borica_rrn,
        $borica_int_ref,
        $borica_pares_status,
        $borica_eci,
        $borica_timestamp,
        $borica_nonce
    ) {
        $pub_key = self::getPublicKeyFromConfig();
        if ('' !== $borica_action) {
            $borica_action_data = mb_strlen($borica_action) . $borica_action;
        } else {
            $borica_action_data = '-';
        }
        if ('' !== $borica_rc) {
            $borica_rc_data = mb_strlen($borica_rc) . $borica_rc;
        } else {
            $borica_rc_data = '-';
        }
        if ('' !== $borica_approval) {
            $borica_approval_data = mb_strlen($borica_approval) . $borica_approval;
        } else {
            $borica_approval_data = '-';
        }
        if ('' !== $borica_terminal) {
            $borica_terminal_data = mb_strlen($borica_terminal) . $borica_terminal;
        } else {
            $borica_terminal_data = '-';
        }
        if ('' !== $borica_trtype) {
            $borica_trtype_data = mb_strlen($borica_trtype) . $borica_trtype;
        } else {
            $borica_trtype_data = '-';
        }
        if ('' !== $borica_amount) {
            $borica_amount_data = mb_strlen($borica_amount) . $borica_amount;
        } else {
            $borica_amount_data = '-';
        }
        if ('' !== $borica_currency) {
            $borica_currency_data = mb_strlen($borica_currency) . $borica_currency;
        } else {
            $borica_currency_data = '-';
        }
        if ('' !== $borica_order) {
            $borica_order_data = mb_strlen($borica_order) . $borica_order;
        } else {
            $borica_order_data = '-';
        }
        if ('' !== $borica_rrn) {
            $borica_rrn_data = mb_strlen($borica_rrn) . $borica_rrn;
        } else {
            $borica_rrn_data = '-';
        }
        if ('' !== $borica_int_ref) {
            $borica_int_ref_data = mb_strlen($borica_int_ref) . $borica_int_ref;
        } else {
            $borica_int_ref_data = '-';
        }
        if ('' !== $borica_pares_status) {
            $borica_pares_status_data = mb_strlen($borica_pares_status) . $borica_pares_status;
        } else {
            $borica_pares_status_data = '-';
        }
        if ('' !== $borica_eci) {
            $borica_eci_data = mb_strlen($borica_eci) . $borica_eci;
        } else {
            $borica_eci_data = '-';
        }
        if ('' !== $borica_timestamp) {
            $borica_timestamp_data = mb_strlen($borica_timestamp) . $borica_timestamp;
        } else {
            $borica_timestamp_data = '-';
        }
        if ('' !== $borica_nonce) {
            $borica_nonce_data = mb_strlen($borica_nonce) . $borica_nonce;
        } else {
            $borica_nonce_data = '-';
        }
        $borica_rfu_data = '-';
        $data =
            $borica_action_data .
            $borica_rc_data .
            $borica_approval_data .
            $borica_terminal_data .
            $borica_trtype_data .
            $borica_amount_data .
            $borica_currency_data .
            $borica_order_data .
            $borica_rrn_data .
            $borica_int_ref_data .
            $borica_pares_status_data .
            $borica_eci_data .
            $borica_timestamp_data .
            $borica_nonce_data .
            $borica_rfu_data;

        $borica_p_sign_bin = hex2bin($borica_p_sign);
        if (false !== strpos($pub_key, 'CERTIFICATE')) {
            $pkeyid = openssl_get_publickey($pub_key);
        } else {
            $pkeyid = $pub_key;
        }
        $result = openssl_verify($data, $borica_p_sign_bin, $pkeyid, OPENSSL_ALGO_SHA256);

        if (1 === $result) {
            return true;
        } elseif (0 === $result) {
            return false;
        } else {
            return false;
        }
    }

    /**
     * Signs payment cancellation/reversal request data for BORICA payment gateway.
     *
     * This method generates a digital signature (P_SIGN) for payment cancellation/reversal
     * by signing the concatenated transaction parameters using the appropriate private key
     * based on currency and test/production mode. Used for reversing or canceling
     * previously authorized payments.
     *
     * @param string $borica_terminal Terminal ID from BORICA configuration
     * @param string|int $borica_trtype Transaction type (typically for cancellation/reversal)
     * @param string|float $borica_amount Amount to be cancelled/reversed
     * @param string $borica_currency Currency code (BGN or EUR) - determines which key to use
     * @param string $borica_order Order number/identifier
     * @param string $borica_timestamp Transaction timestamp
     * @param string $borica_nonce Unique nonce value for the transaction
     * @return array Array containing:
     *               - 'boricaError' (string): Empty string on success, error message on failure
     *               - 'pSign' (string): Hexadecimal signature in uppercase, empty string on error
     */
    public static function signDropPayment(
        $borica_terminal,
        $borica_trtype,
        $borica_amount,
        $borica_currency,
        $borica_order,
        $borica_timestamp,
        $borica_nonce
    ) {
        $result = [];
        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        if ('BGN' === $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_bgn');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_bgn');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_bgn');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_bgn');
        }
        if ('EUR' === $borica_currency) {
            $borica_test_key = (string) Configuration::get('mtborica_test_key_eur');
            $borica_production_key = (string) Configuration::get('mtborica_production_key_eur');
            $borica_test_password = (string) Configuration::get('mtborica_test_password_eur');
            $borica_production_password = (string) Configuration::get('mtborica_production_password_eur');
        }
        if (1 === $borica_testmode) {
            $priv_key = $borica_test_key;
            $priv_key_password = $borica_test_password;
        } else {
            $priv_key = $borica_production_key;
            $priv_key_password = $borica_production_password;
        }
        $data =
            strlen($borica_terminal) . $borica_terminal .
            strlen($borica_trtype) . $borica_trtype .
            strlen($borica_amount) . $borica_amount .
            strlen($borica_currency) . $borica_currency .
            strlen($borica_order) . $borica_order .
            strlen($borica_timestamp) . $borica_timestamp .
            strlen($borica_nonce) . $borica_nonce .
            '-';
        try {
            $pkeyid = openssl_get_privatekey($priv_key, $priv_key_password);
            openssl_sign($data, $signature, $pkeyid, OPENSSL_ALGO_SHA256);
            $result['boricaError'] = '';
            $result['pSign'] = strtoupper(bin2hex($signature));
        } catch (Exception $e) {
            // Log detailed error for debugging (server-side only)
            if (class_exists('MtboricaLogger')) {
                MtboricaLogger::error('BORICA signDropPayment error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
            }
            // Return generic error message (no sensitive information exposed)
            $result['boricaError'] = 'Signature generation failed';
            $result['pSign'] = '';
        }
        return $result;
    }

    /**
     * Recreate cart from order and redirect customer to cart
     *
     * @param int $order_id Order ID
     * @return bool True on success, false on failure
     */
    public static function recreateCart($order_id)
    {
        if (!$order_id) {
            return false;
        }

        try {
            /** @var Context $context */
            $context = Context::getContext();

            /** @var Order $order */
            $order = new Order((int) $order_id);
            if (!Validate::isLoadedObject($order)) {
                return false;
            }

            // Get customer - must be logged in to recreate cart
            if (!$context->customer->isLogged() || $context->customer->id != $order->id_customer) {
                // If customer is not logged in or doesn't match, redirect to login
                Tools::redirect($context->link->getPageLink('authentication', true, null, [
                    'back' => $context->link->getPageLink('order', true)
                ]));
                return false;
            }

            // Get order details (products)
            /** @var array $order_details */
            $order_details = OrderDetail::getList($order_id);
            if (empty($order_details)) {
                return false;
            }

            // Create new cart
            /** @var Cart $cart */
            $cart = new Cart();
            $cart->id_customer = (int) $context->customer->id;
            $cart->id_currency = (int) $order->id_currency;
            $cart->id_lang = (int) $context->language->id;
            $cart->id_address_delivery = (int) $order->id_address_delivery;
            $cart->id_address_invoice = (int) $order->id_address_invoice;
            $cart->id_shop = (int) $context->shop->id;
            $cart->secure_key = $context->customer->secure_key;

            if (!$cart->add()) {
                return false;
            }

            // Add products from order to cart
            foreach ($order_details as $order_detail) {
                $product_id = (int) $order_detail['product_id'];
                $product_attribute_id = (int) $order_detail['product_attribute_id'];
                $quantity = (int) $order_detail['product_quantity'];

                // Check if product still exists and is available
                /** @var Product $product */
                $product = new Product($product_id);
                if (!Validate::isLoadedObject($product) || !$product->active) {
                    continue; // Skip unavailable products
                }

                // Add product to cart
                $result = $cart->updateQty(
                    $quantity,
                    $product_id,
                    $product_attribute_id,
                    false, // no customization
                    'up', // direction
                    0, // id_customization
                    null, // shop
                    false // dontUpdateCart
                );

                if ($result < 0) {
                    // Silently fail and continue
                }
            }

            // Update cart delivery address
            $cart->updateAddressId(
                (int) $order->id_address_delivery,
                (int) $order->id_address_delivery
            );

            // Save cart
            $cart->update();

            // Set cart as current cart in context
            $context->cart = $cart;
            $context->cookie->id_cart = (int) $cart->id;
            $context->cookie->write();

            // Redirect to cart page
            Tools::redirect($context->link->getPageLink('cart', true));

            return true;
        } catch (Exception $e) {
            // Log detailed error for debugging (server-side only)
            if (class_exists('MtboricaLogger')) {
                MtboricaLogger::error('BORICA recreateCart error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine(),
                    'order_id' => $order_id
                ]);
            }
            // Return false on error (no sensitive information exposed)
            return false;
        }
    }

    /**
     * Calculates the individual recurring payment amount based on total order amount,
     * frequency, and duration settings.
     *
     * This function normalizes both the duration and frequency into months, and calculates
     * the number of payments expected during the period. It then divides the total amount
     * by this number to obtain the amount per installment.
     *
     * @since 2.0.3
     *
     * @param float  $total_amount         The total order amount.
     * @param string $duration             The duration value (e.g. 12).
     * @param string $duration_unit        The unit for duration ('D', 'M', 'Q', 'H', 'Y').
     * @param string $frequency            The frequency of payment (e.g. every 1 month).
     * @param string $frequency_unit       The unit for frequency ('D', 'M', 'Q', 'H', 'Y').
     *
     * @return array|null Returns array with 'amount' and 'payments' keys, or null on error.
     */
    public static function calculateFirstPaymentAmount(
        $total_amount,
        $duration,
        $duration_unit,
        $frequency,
        $frequency_unit
    ) {
        $duration_int = (int) $duration;
        $frequency_int = (int) $frequency;

        if ($duration_int <= 0 || $frequency_int <= 0) {
            return null;
        }

        $duration_days = self::convertToDays($duration_int, $duration_unit);
        $frequency_days = self::convertToDays($frequency_int, $frequency_unit);

        if ($duration_days === 0 || $frequency_days === 0) {
            return null;
        }

        $total_payments = (int) floor($duration_days / $frequency_days);

        if ($total_payments < 1) {
            return null;
        }

        return [
            'amount' => round($total_amount / $total_payments, 2),
            'payments' => $total_payments
        ];
    }

    /**
     * Converts a duration value with a specific unit to days.
     *
     * This method is used to normalize different duration units (days, months, quarters, half-years, years)
     * into a common unit (days) in order to easily compare or calculate recurring payment periods.
     *
     * Conversion rates:
     * - D (Day)    -> 1 day
     * - M (Month)  -> 30 days (approximate)
     * - Q (Quarter)-> 90 days (approximate)
     * - H (Half-Year) -> 180 days (approximate)
     * - Y (Year)   -> 365 days (approximate)
     *
     * @since 2.x
     *
     * @param int    $value The numeric duration or frequency value to convert.
     * @param string $unit  The unit of the provided value. Allowed values: D, M, Q, H, Y.
     *
     * @return int The equivalent duration in days.
     */
    public static function convertToDays($value, $unit)
    {
        switch ($unit) {
            case 'D':
                return $value;
            case 'M':
                return $value * 30;
            case 'Q':
                return $value * 90;
            case 'H':
                return $value * 180;
            case 'Y':
                return $value * 365;
            default:
                return $value; // fallback, should not occur with validated input
        }
    }

    /**
     * Calculates the recurring payment expiration date (`recur_exp`) based on duration and unit.
     *
     * This function takes the number of payment intervals (duration), a unit of time (day, month, quarter, half-year, year),
     * and an optional start date, and calculates the end date of the last installment period.
     * The result is formatted as a string in `YYYYMMDD` format for use in BORICA requests.
     *
     * Example:
     * - Start date: 2025-04-07
     * - Duration: 12
     * - Unit: M (month)
     * - Result: 20260307 (the 12th installment ends on March 7, 2026)
     *
     * Supported units:
     * - `D`: Days
     * - `M`: Months
     * - `Q`: Quarters (3 months)
     * - `H`: Half-years (6 months)
     * - `Y`: Years
     *
     * @since 2.0.3
     *
     * @param string $recur_duration      Number of periods (e.g., 12 for 12 months).
     * @param string $recur_duration_unit Unit of the duration (D, M, Q, H, Y).
     * @param string $start_date          Optional start date in format `Y-m-d`. Defaults to current date if not provided.
     *
     * @return string The calculated expiration date in format `YYYYMMDD`, or empty string on failure.
     */
    public static function calculateRecurExp(
        string $recur_duration,
        string $recur_duration_unit,
        string $recur_freq,
        string $recur_freq_unit,
        string $recur_mday_payment,
        string $start_date = ''
    ): string {
        if (!is_numeric($recur_duration) || (int) $recur_duration <= 0) {
            return '';
        }

        $duration_days = self::convertToDays($recur_duration, $recur_duration_unit);
        $frequency_days = self::convertToDays($recur_freq, $recur_freq_unit);
        $total_payments = (int) floor($duration_days / $frequency_days);

        try {
            $duration = (int) $recur_duration;
            $freq = (int) $recur_freq;
            $start = !empty($start_date) ? new DateTime($start_date) : new DateTime('now');

            if ($recur_freq_unit === 'D') {
                $last = (clone $start)->add(new DateInterval("P" . ($freq * ($total_payments - 1)) . "D"));
                return $last->format('Ymd');
            }

            $dates = [$start];

            switch ($recur_duration_unit) {
                case 'D':
                    $end = (clone $start)->add(new DateInterval("P{$duration}D"));
                    break;
                case 'M':
                    if ($recur_freq_unit === 'M') {
                        if ($duration % $freq === 0) {
                            $adjusted_duration = $duration;
                        } else {
                            $adjusted_duration = $duration - 1;
                        }
                        $end = (clone $start)->add(new DateInterval("P{$adjusted_duration}M"));
                    } else {
                        $end = (clone $start)->add(new DateInterval("P{$duration}M"));
                    }
                    break;
                case 'Q':
                    $end = (clone $start)->add(new DateInterval("P" . ($duration * 3) . "M"));
                    break;
                case 'H':
                    $end = (clone $start)->add(new DateInterval("P" . ($duration * 6) . "M"));
                    break;
                case 'Y':
                    $end = (clone $start)->add(new DateInterval("P{$duration}Y"));
                    break;
                default:
                    return '';
            }

            $index = 1;
            while (true) {
                $next = clone $start;

                switch ($recur_freq_unit) {
                    case 'M':
                        $next->add(new DateInterval("P" . ($freq * $index) . "M"));
                        break;
                    case 'Q':
                        $next->add(new DateInterval("P" . ($freq * $index * 3) . "M"));
                        break;
                    case 'H':
                        $next->add(new DateInterval("P" . ($freq * $index * 6) . "M"));
                        break;
                    case 'Y':
                        $next->add(new DateInterval("P" . ($freq * $index) . "Y"));
                        break;
                    default:
                        return end($dates)->format('Ymd');
                }

                if (
                    in_array($recur_freq_unit, ['M', 'Q', 'H', 'Y']) &&
                    !empty($recur_mday_payment) &&
                    is_numeric($recur_mday_payment)
                ) {
                    $desired_day = (int) $recur_mday_payment;
                    $last_day = (int) date('t', $next->getTimestamp());
                    $adjusted_day = ($desired_day === 31) ? $last_day : min($desired_day, $last_day);

                    $next->setDate(
                        (int) $next->format('Y'),
                        (int) $next->format('m'),
                        $adjusted_day
                    );
                }

                if ($next >= $end || $index >= $total_payments) {
                    break;
                }

                $dates[] = $next;
                $index++;
            }

            return end($dates)->format('Ymd');
        } catch (Exception $e) {
            // Log detailed error for debugging (server-side only)
            if (class_exists('MtboricaLogger')) {
                MtboricaLogger::error('BORICA calculateRecurExp error', [
                    'message' => $e->getMessage(),
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                ]);
            }
            // Return empty string on error (no sensitive information exposed)
            return '';
        }
    }
}
