<?php

/**
 * Class MtboricaLogger
 *
 * Logger class for BORICA payment gateway operations.
 * Uses PrestaShop's FileLogger for logging with automatic rotation.
 *
 * This class uses static methods for convenience, but internally uses a singleton pattern
 * to manage the FileLogger instance. The logger automatically rotates logs monthly.
 *
 * Usage (recommended - static methods):
 *   MtboricaLogger::log('info', 'Message', $context);
 *   MtboricaLogger::error('Error message', $context);
 *   MtboricaLogger::logPaymentSent($paymentData);
 *
 * Usage (instance methods - if needed for dependency injection):
 *   $logger = new MtboricaLogger();
 *   $logger->logMessage('info', 'Message', $context);
 */
class MtboricaLogger
{
    /**
     * @var FileLogger Logger instance
     */
    private $logger;

    /**
     * @var string Log filename pattern
     */
    private $logFilename;

    /**
     * @var MtboricaLogger Singleton instance for static methods
     */
    private static $instance = null;

    /**
     * Constructor
     *
     * @param string|null $logFilename Custom log filename (optional)
     */
    public function __construct($logFilename = null)
    {
        $this->logger = new FileLogger();

        if ($logFilename === null) {
            // Default: monthly rotation (mtborica_YYYY-MM.log)
            $this->logFilename = _PS_ROOT_DIR_ . '/var/logs/mtborica_' . date('Y-m') . '.log';
        } else {
            $this->logFilename = $logFilename;
        }

        $this->logger->setFilename($this->logFilename);
    }

    /**
     * Get singleton instance for static method calls
     *
     * @return MtboricaLogger
     */
    private static function getInstance()
    {
        if (self::$instance === null) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Internal method: Log message with level and context
     *
     * @param string $level Log level: 'error', 'warning', 'info', 'debug'
     * @param string $message Log message
     * @param array $context Additional context data
     * @return void
     */
    private function logMessage($level, $message, array $context = [])
    {
        // Build log message with context
        $log_message = $message;
        if (!empty($context)) {
            $log_message .= ' | Context: ' . json_encode($context, JSON_UNESCAPED_UNICODE);
        }

        // Log according to level
        switch (strtolower($level)) {
            case 'error':
                $this->logger->logError($log_message);
                break;
            case 'warning':
                $this->logger->logWarning($log_message);
                break;
            case 'info':
                $this->logger->logInfo($log_message);
                break;
            case 'debug':
                // FileLogger doesn't have logDebug, use logInfo for debug
                if (defined('_PS_MODE_DEV_') && _PS_MODE_DEV_) {
                    $this->logger->logInfo('[DEBUG] ' . $log_message);
                }
                break;
            default:
                $this->logger->logInfo($log_message);
        }
    }

    // ========== Static methods (recommended usage) ==========

    /**
     * Static method: Log message with level and context
     *
     * @param string $level Log level: 'error', 'warning', 'info', 'debug'
     * @param string $message Log message
     * @param array $context Additional context data
     * @return void
     */
    public static function log($level, $message, array $context = [])
    {
        self::getInstance()->logMessage($level, $message, $context);
    }

    /**
     * Static method: Log error
     *
     * @param string $message Error message
     * @param array $context Additional context
     * @return void
     */
    public static function error($message, array $context = [])
    {
        self::log('error', $message, $context);
    }

    /**
     * Static method: Log warning
     *
     * @param string $message Warning message
     * @param array $context Additional context
     * @return void
     */
    public static function warning($message, array $context = [])
    {
        self::log('warning', $message, $context);
    }

    /**
     * Static method: Log info
     *
     * @param string $message Info message
     * @param array $context Additional context
     * @return void
     */
    public static function info($message, array $context = [])
    {
        self::log('info', $message, $context);
    }

    /**
     * Static method: Log debug (only in dev mode)
     *
     * @param string $message Debug message
     * @param array $context Additional context
     * @return void
     */
    public static function debug($message, array $context = [])
    {
        self::log('debug', $message, $context);
    }

    /**
     * Static method: Log payment sent to BORICA
     *
     * @param array $data Payment data
     * @return void
     */
    public static function logPaymentSent(array $data)
    {
        self::log('info', '1.mtborica.borica_send', $data);
    }

    /**
     * Static method: Log payment callback/response from BORICA
     *
     * @param array $data Response data from BORICA
     * @return void
     */
    public static function logPaymentReceived(array $data)
    {
        $level = ($data['RC'] ?? '') === '00' ? 'info' : 'warning';
        self::log($level, '2.mtborica.borica_received', $data);
    }

    /**
     * Static method: Log check payment sent to BORICA
     *
     * @param array $data Payment data
     * @return void
     */
    public static function logCheckPaymentSent(array $data)
    {
        self::log('info', '3.mtborica.borica_check_payment_sent', $data);
    }

    /**
     * Static method: Log check payment received from BORICA
     *
     * @param array $data Response data from BORICA
     * @return void
     */
    public static function logCheckPaymentReceived(array $data)
    {
        self::log('info', '4.mtborica.borica_check_payment_received', $data);
    }

    /**
     * Static method: Log drop payment sent to BORICA
     *
     * @param array $data Payment data
     * @return void
     */
    public static function logDropPaymentSent(array $data)
    {
        self::log('info', '5.mtborica.borica_drop_payment_sent', $data);
    }

    /**
     * Static method: Log drop payment received from BORICA
     *
     * @param array $data Response data from BORICA
     * @return void
     */
    public static function logDropPaymentReceived(array $data)
    {
        self::log('info', '6.mtborica.borica_drop_payment_received', $data);
    }

    /**
     * Static method: Log cancel recurring payment sent to BORICA
     *
     * @param array $data Payment data
     * @return void
     */
    public static function logCancelRecurringPaymentSent(array $data)
    {
        self::log('info', '7.mtborica.borica_cancel_recurring_payment_sent', $data);
    }

    /**
     * Static method: Log cancel recurring payment received from BORICA
     *
     * @param array $data Response data from BORICA
     * @return void
     */
    public static function logCancelRecurringPaymentReceived(array $data)
    {
        self::log('info', '8.mtborica.borica_cancel_recurring_payment_received', $data);
    }
    // ========== Instance methods (for dependency injection if needed) ==========

    /**
     * Instance method: Log message with level and context
     *
     * @param string $level Log level
     * @param string $message Log message
     * @param array $context Additional context
     * @return void
     */
    public function logMessageInstance($level, $message, array $context = [])
    {
        $this->logMessage($level, $message, $context);
    }

    /**
     * Clean old log files (older than specified days)
     *
     * This method should be called periodically (e.g., via cron or during module operations)
     * to prevent log files from growing indefinitely.
     *
     * @param int $days Number of days to keep logs (default: 90 days)
     * @return int Number of deleted files
     */
    public static function cleanOldLogs($days = 90)
    {
        $log_dir = _PS_ROOT_DIR_ . '/var/logs/';
        $pattern = $log_dir . 'mtborica_*.log';
        $files = glob($pattern);
        $deleted = 0;
        $cutoff_time = time() - ($days * 24 * 60 * 60);

        if ($files) {
            foreach ($files as $file) {
                if (file_exists($file) && filemtime($file) < $cutoff_time) {
                    if (@unlink($file)) {
                        $deleted++;
                    }
                }
            }
        }

        return $deleted;
    }
}
