<?php
/**
 * Class MtboricaOrder
 *
 * ObjectModel representing order data sent to BORICA bank stored in
 * database table `mtborica_orders`.
 *
 * Fields map:
 * - id (int): Primary key
 * - action (string): Action code (max 20)
 * - rc (string): Response code (max 20)
 * - created_at (string): Creation timestamp (Y-m-d H:i:s)
 * - increment_id (string|null): Order increment ID (max 50)
 * - status (string): Order status (max 255)
 * - rrn (string): Retrieval Reference Number (max 20)
 * - int_ref (string): Internal reference (max 20)
 * - approval (string): Approval code (max 6)
 * - merch_gmt (string): Merchant GMT offset (max 3)
 * - request_cancel (string): Cancel request (max 20)
 * - cancel_amount (string): Cancel amount (max 20)
 * - nonce (string): Nonce value (max 32)
 * - merch_rn_id (string): Merchant RN ID (max 16)
 * - recur_id (string): Recurring ID (max 9)
 * - amount (string): Order amount (max 20)
 *
 * @property int         $id
 * @property string      $action
 * @property string      $rc
 * @property string      $created_at
 * @property string|null $increment_id
 * @property string      $status
 * @property string      $rrn
 * @property string      $int_ref
 * @property string      $approval
 * @property string      $merch_gmt
 * @property string      $request_cancel
 * @property string      $cancel_amount
 * @property string      $nonce
 * @property string      $merch_rn_id
 * @property string      $recur_id
 * @property string      $amount
 */
class MtboricaOrder extends ObjectModel
{
    public $id;
    public $action;
    public $rc;
    public $created_at;
    public $increment_id;
    public $status;
    public $rrn;
    public $int_ref;
    public $approval;
    public $merch_gmt;
    public $request_cancel;
    public $cancel_amount;
    public $nonce;
    public $merch_rn_id;
    public $recur_id;
    public $amount;

    /**
     * @see ObjectModel::$definition
     */
    public static $definition = array(
        'table' => 'mtborica_orders',
        'primary' => 'id',
        'fields' => array(
            'action' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 20),
            'rc' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 20),
            'created_at' => array('type' => self::TYPE_DATE, 'validate' => 'isDateFormat', 'required' => true),
            'increment_id' => array('type' => self::TYPE_STRING, 'required' => false, 'size' => 50),
            'status' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 255),
            'rrn' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => false, 'size' => 20),
            'int_ref' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => false, 'size' => 20),
            'approval' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => false, 'size' => 6),
            'merch_gmt' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 3),
            'request_cancel' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 20),
            'cancel_amount' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 20),
            'nonce' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 32),
            'merch_rn_id' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => false, 'size' => 16),
            'recur_id' => array('type' => self::TYPE_STRING, 'required' => false, 'size' => 9),
            'amount' => array('type' => self::TYPE_STRING, 'validate' => 'isString', 'required' => true, 'size' => 20),
        ),
    );

    /**
     * Create a new order record with provided data
     *
     * @param array $data Order data array
     * @return MtboricaOrder|false Created order object or false on failure
     */
    public static function createOrder($data)
    {
        $order = new self();

        // Set all fields from data array
        foreach ($data as $field => $value) {
            if (property_exists($order, $field)) {
                // Handle null/empty values for optional fields
                if (($value === null || $value === '') && in_array($field, ['increment_id', 'recur_id', 'rrn', 'int_ref', 'approval'])) {
                    $order->$field = '';
                } else {
                    $order->$field = (string) $value;
                }
            }
        }

        // Set created_at if not provided
        if (empty($order->created_at)) {
            $order->created_at = date('Y-m-d H:i:s');
        }

        if ($order->save()) {
            return $order;
        }

        return false;
    }

    /**
     * Get order by increment ID
     *
     * @param string $increment_id
     * @return MtboricaOrder|false
     */
    public static function getByIncrementId($increment_id)
    {
        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'mtborica_orders` WHERE `increment_id` = \'' . pSQL($increment_id) . '\'';
        $result = Db::getInstance()->getRow($sql);

        if (!$result) {
            return false;
        }

        $order = new self();
        $order->hydrate($result);
        return $order;
    }

    /**
     * Get order by nonce
     *
     * @param string $nonce
     * @return MtboricaOrder|false
     */
    public static function getByNonce($nonce)
    {
        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'mtborica_orders` WHERE `nonce` = \'' . pSQL($nonce) . '\' ORDER BY `created_at` DESC';
        $result = Db::getInstance()->getRow($sql);

        if (!$result) {
            return false;
        }

        $order = new self();
        $order->hydrate($result);
        return $order;
    }

    /**
     * Get all orders
     *
     * @param int $limit Limit number of results
     * @param int $offset Offset for pagination
     * @return array
     */
    public static function getAll($limit = 0, $offset = 0)
    {
        $sql = 'SELECT * FROM `' . _DB_PREFIX_ . 'mtborica_orders` ORDER BY `created_at` DESC';

        if ($limit > 0) {
            $sql .= ' LIMIT ' . (int) $offset . ', ' . (int) $limit;
        }

        $result = Db::getInstance()->executeS($sql);

        if (!$result) {
            return array();
        }

        return $result;
    }

    /**
     * Update order status
     *
     * @param string $increment_id
     * @param string $status
     * @param array $additional_data Additional fields to update
     * @return bool
     */
    public static function updateOrderStatus($increment_id, $status, $additional_data = array())
    {
        $order = self::getByIncrementId($increment_id);
        if (!$order) {
            return false;
        }

        $order->status = $status;

        // Update additional fields if provided
        foreach ($additional_data as $field => $value) {
            if (property_exists($order, $field)) {
                $order->$field = $value;
            }
        }

        return $order->save();
    }

    /**
     * Update Borica order by nonce
     *
     * @param array $data Order data array containing 'nonce' as key
     * @return bool
     */
    public static function updateBoricaOrder($data)
    {
        if (!isset($data['nonce']) || empty($data['nonce'])) {
            return false;
        }

        $order = self::getByNonce($data['nonce']);
        if (!$order) {
            return false;
        }

        // Update all fields from data array (except nonce, which is the key)
        foreach ($data as $field => $value) {
            if ($field !== 'nonce' && property_exists($order, $field)) {
                // Handle null/empty values for optional fields
                if (($value === null || $value === '') && in_array($field, ['increment_id', 'recur_id', 'merch_rn_id', 'rrn', 'int_ref', 'approval'])) {
                    $order->$field = '';
                } else {
                    $order->$field = (string) $value;
                }
            }
        }

        return $order->save();
    }

    /**
     * Get orders count
     *
     * @param string $status Filter by status (optional)
     * @return int
     */
    public static function getCount($status = null)
    {
        $sql = 'SELECT COUNT(*) FROM `' . _DB_PREFIX_ . 'mtborica_orders`';

        if ($status !== null) {
            $sql .= ' WHERE `status` = \'' . pSQL($status) . '\'';
        }

        return (int) Db::getInstance()->getValue($sql);
    }

    /**
     * Delete orders older than specified days
     *
     * @param int $days Number of days
     * @return bool
     */
    public static function deleteOldOrders($days = 90)
    {
        $sql = 'DELETE FROM `' . _DB_PREFIX_ . 'mtborica_orders` WHERE `created_at` < DATE_SUB(NOW(), INTERVAL ' . (int) $days . ' DAY)';
        return Db::getInstance()->execute($sql);
    }
}
