<?php
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaOrder.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaBoricaHelper.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaRecurring.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaLogger.php';

class MtboricaValidationModuleFrontController extends ModuleFrontController
{
    /**
     * BORICA Transaction Type: Authorization
     *
     * This constant represents the transaction type for authorization within the BORICA payment gateway system.
     *
     * The value `1` indicates an authorization transaction, where funds are reserved on the cardholder's account
     * but not yet captured. This is typically used in scenarios where payment is confirmed but the final amount
     * may change before the capture.
     *
     * @var int
     */
    private const BORICA_TRTYPE_AUTHORIZATION = 1;

    /**
     * BORICA Country Code
     *
     * This constant defines the country code used in transactions with the BORICA payment gateway.
     *
     * The value `'BG'` represents Bulgaria, which is the default country for transactions processed
     * through BORICA. This code is used in various parts of the payment request to indicate the origin
     * of the transaction.
     *
     * Usage:
     * - This constant is included in the payment gateway parameters sent to BORICA to specify the
     *   country where the transaction is initiated.
     *
     * @var string BORICA_COUNTRY
     */
    private const BORICA_COUNTRY = 'BG';

    /**
     * BORICA Addendum Fields
     *
     * This constant defines additional data fields to be included in the BORICA payment request.
     *
     * The value `'AD,TD'` represents specific addendum fields that provide extra transaction details
     * or options. These fields can include additional data such as merchant details or transaction
     * specifics required by BORICA for processing the payment.
     *
     * Usage:
     * - This constant is used when constructing the payment request parameters to ensure that the necessary
     *   addendum data is included in the transaction sent to BORICA.
     *
     * @var string BORICA_ADDENDUM
     */
    private const BORICA_ADDENDUM = 'AD,TD';

    /**
     * Processes the payment order and redirects to bank payment form
     */
    public function postProcess()
    {
        // Check if cart exists and is valid
        if (!$this->context->cart || !$this->context->cart->id) {
            Tools::redirect('index.php?controller=order');
        }

        // Check if customer is logged in
        if (!$this->context->customer->isLogged()) {
            Tools::redirect('index.php?controller=authentication&back=' . urlencode('index.php?controller=order'));
        }

        // Validate cart
        if (!$this->context->cart->checkQuantities()) {
            Tools::redirect('index.php?controller=order');
        }

        // Keep cart recurring metadata in sync before further checks
        if (Mtborica::isRecurringEnabled()) {
            Mtborica::cleanupOrphanedCartPlans((int) $this->context->cart->id);
        }

        // Check if this is a recurring payment
        // First, try to get from hidden field (faster and more direct)
        // This field is set in hookPaymentOptions when recurring payment is selected
        $recurring_plan_id_from_form = (int) Tools::getValue('mtborica_recurring_plan_id', 0);
        $is_recurring_payment = false;

        if (Mtborica::isRecurringEnabled()) {
            if ($recurring_plan_id_from_form > 0) {
                // Use form field (faster, more direct - already validated in hookPaymentOptions)
                $is_recurring_payment = true;
            } else {
                // No form field - check cart directly (fallback for edge cases)
                $recurring_plan_id_from_cart = Mtborica::cartHasSingleRecurringPlan($this->context->cart);
                $is_recurring_payment = ($recurring_plan_id_from_cart > 0);
            }
        }

        // Set payment method name based on payment type
        $payment_method_name = $this->module->displayName;
        if ($is_recurring_payment) {
            $payment_method_name = $this->l('Recurring Payment by Credit/Debit Card');
        }

        // Create the order
        $this->module->validateOrder(
            $this->context->cart->id,
            Configuration::get('PS_OS_PREPARATION'),
            $this->context->cart->getOrderTotal(),
            $payment_method_name,
            null,
            [],
            $this->context->currency->id,
            false,
            $this->context->customer->secure_key
        );

        // Get the created order
        $order_id = $this->module->currentOrder;
        $order = new Order($order_id);

        // Prepare bank payment data
        $borica_payment_data = $this->prepareBankPaymentData($order, $is_recurring_payment);
        $json = json_encode($borica_payment_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        $token = strtoupper(Tools::passwdGen(40));
        Db::getInstance()->insert('mtborica_payloads', [
            'token' => pSQL($token),
            'payload_json' => pSQL($json),
            'date_add' => date('Y-m-d H:i:s'),
            'date_expire' => date('Y-m-d H:i:s', strtotime('+1 day')),
        ]);

        if (Mtborica::isRecurringEnabled()) {
            // Remove temporary cart plan records now that order is created
            Mtborica::deleteCartRecurringPlans((int) $this->context->cart->id);
            // Clean up stale plan records for abandoned carts (older than 1 day)
            Mtborica::cleanupOrphanedCartPlanRecords(1);
        }

        $redirect_url = $this->context->link->getPageLink(
            'order-confirmation',
            true,
            (int) $this->context->language->id,
            [
                'id_cart' => (int) $this->context->cart->id,
                'id_module' => (int) $this->module->id,
                'id_order' => (int) $this->module->currentOrder,
                'key' => $this->context->customer->secure_key,
                'borica_token' => $token,
            ]
        );
        Tools::redirect($redirect_url);
    }

    /**
     * Prepares the data for bank payment
     */
    private function prepareBankPaymentData($order, $is_recurring_payment)
    {
        $result = [];
        $order_id = $order->id;
        // Get customer data from order
        $customer = new Customer($order->id_customer);
        $address = new Address($order->id_address_delivery);

        $cardholder_email_address = $customer->email;
        $cardholder_home_phone = $address->phone;
        $borica_firstname = $customer->firstname;
        $transliterated_borica_firstname = $this->transliterate($borica_firstname);
        $cleaned_borica_firstname = preg_replace('/[^a-zA-Z\-\.\ ]/', '', $transliterated_borica_firstname);
        $trimmed_borica_firstname = mb_substr($cleaned_borica_firstname, 0, 22);
        $uppercase_borica_firstname = strtoupper($trimmed_borica_firstname);
        $borica_lastname = $customer->lastname;
        $transliterated_borica_lastname = $this->transliterate($borica_lastname);
        $cleaned_borica_lastname = preg_replace('/[^a-zA-Z\-\.\ ]/', '', $transliterated_borica_lastname);
        $trimmed_borica_lastname = mb_substr($cleaned_borica_lastname, 0, 22);
        $uppercase_borica_lastname = strtoupper($trimmed_borica_lastname);
        $cardholder_name = $uppercase_borica_firstname . ' ' . $uppercase_borica_lastname;
        $borica_currency = $this->context->currency->iso_code;
        $borica_terminal = '';
        $borica_merchant = '';
        if ('BGN' === $borica_currency) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_bgn');
            $borica_merchant = (string) Configuration::get('mtborica_mid_bgn');
        }
        if ('EUR' === $borica_currency) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_eur');
            $borica_merchant = (string) Configuration::get('mtborica_mid_eur');
        }
        $borica_trtype = self::BORICA_TRTYPE_AUTHORIZATION;
        $borica_amount = number_format((float) $order->total_paid, 2, '.', '');

        if (strlen($order_id) >= 6) {
            $borica_order = substr($order_id, -6);
        } else {
            $borica_order = str_pad($order_id, 6, '0', STR_PAD_LEFT);
        }
        $base_url = rtrim(Tools::getShopDomainSsl(true, true), '/');
        $borica_desc = $this->l('Order of goods from') . ' ' . $base_url;
        $borica_merch_name = (string) Configuration::get('mtborica_mname');
        $borica_email = (string) Configuration::get('mtborica_email');
        $borica_country = self::BORICA_COUNTRY;
        $original_date = $order->date_add;
        $date = new \DateTime($original_date);
        $borica_merch_gmt =
            isset(explode(':', $date->format('P'))[0]) ?
            explode(':', $date->format('P'))[0] :
            '+03';
        $borica_addendum = self::BORICA_ADDENDUM;
        $borica_ad_cust_bor_order_id = $borica_order . $order_id;
        $borica_timestamp = gmdate('YmdHis');
        $borica_nonce = strtoupper(bin2hex(random_bytes(16)));
        $borica_lang = (string) Configuration::get('mtborica_payment_lang');

        $recur_mday_payment = '';
        $recur_id = 0;
        $merchRnId = '';
        if ($is_recurring_payment) {
            $recur_id = (string) Tools::getValue('mtborica_recurring_plan_id', 0);
            $recurring_plan = MtboricaRecurring::getById($recur_id);
            if ($recurring_plan) {
                $borica_amount = MtboricaBoricaHelper::calculateFirstPaymentAmount(
                    (float) $order->total_paid,
                    $recurring_plan->recur_duration,
                    $recurring_plan->recur_duration_unit,
                    $recurring_plan->recur_freq,
                    $recurring_plan->recur_freq_unit
                );
                $borica_amount = number_format(is_array($borica_amount) ? (float) $borica_amount['amount'] : 0, 2, '.', '');

                $recur_duration = $recurring_plan->recur_duration;
                $recur_duration_unit = $recurring_plan->recur_duration_unit;
                $recur_freq = $recurring_plan->recur_freq;
                $recur_freq_unit = $recurring_plan->recur_freq_unit;
                $recur_mday_payment = $recurring_plan->recur_mday_payment;
                $borica_order_rn = str_pad($order_id, 8, '0', STR_PAD_LEFT);
                $persNameRec = 'RECPAYPS';
                $merchRnId = $persNameRec . $borica_order_rn;
                $recur_exp = MtboricaBoricaHelper::calculateRecurExp(
                    $recur_duration,
                    $recur_duration_unit,
                    $recur_freq,
                    $recur_freq_unit,
                    $recur_mday_payment
                );
                $result += [
                    'RECUR_FREQ' => $recur_freq,
                    'RECUR_EXP' => $recur_exp,
                    'RECUR_FREQ_UNIT' => $recur_freq_unit,
                    'MERCH_TRAN_STATE' => 'IS',
                    'MERCH_RN_ID' => $merchRnId,
                    'RECUR_ID' => $recur_id,
                ];
                if ($recur_mday_payment) {
                    $result += ['RECUR_MDAY_PAYMENT' => $recur_mday_payment];
                }
                $borica_desc = $this->l('Partial payment on order from') . ' ' . $base_url;
            }
        }

        $browser_ip_address = Tools::getRemoteAddr();
        $browser_screen_height = (int) Tools::getValue('browserScreenHeight', 0);
        $browser_screen_width = (int) Tools::getValue('browserScreenWidth', 0);
        $borica_m_info_arr = [
            'browserIP' => $browser_ip_address,
            'browserScreenHeight' => $browser_screen_height,
            'browserScreenWidth' => $browser_screen_width,
            'homePhone' => $cardholder_home_phone,
            'email' => $cardholder_email_address,
            'cardholderName' => $cardholder_name,
        ];
        $borica_m_info_str = json_encode($borica_m_info_arr, JSON_UNESCAPED_UNICODE);

        $date_time = \DateTime::createFromFormat('YmdHis', $borica_timestamp, new \DateTimeZone('UTC'));
        $new_format = 'Y-m-d H:i:s';
        $order_data = [
            'action' => '999',
            'rc' => '999',
            'created_at' => $date_time->format($new_format),
            'increment_id' => $order_id,
            'status' => '999',
            'rrn' => '999',
            'int_ref' => '999',
            'approval' => '999',
            'merch_gmt' => $borica_merch_gmt,
            'request_cancel' => '999',
            'cancel_amount' => '0.00',
            'nonce' => $borica_nonce,
            'merch_rn_id' => $merchRnId,
            'recur_id' => $recur_id,
            'amount' => (string) $borica_amount,
        ];
        MtboricaOrder::createOrder($order_data);

        // Ensure all parameters are strings for signAuthorization
        $borica_p_sign = MtboricaBoricaHelper::signAuthorization(
            (string) $borica_terminal,
            (string) $borica_trtype,
            (string) $borica_amount,
            (string) $borica_currency,
            (string) $borica_order,
            (string) $borica_timestamp,
            (string) $borica_nonce
        );

        $result += [
            'TERMINAL' => $borica_terminal,
            'TRTYPE' => $borica_trtype,
            'AMOUNT' => $borica_amount,
            'CURRENCY' => $borica_currency,
            'ORDER' => $borica_order,
            'DESC' => $borica_desc,
            'MERCHANT' => $borica_merchant,
            'MERCH_NAME' => $borica_merch_name,
            'MERCH_URL' => $base_url,
            'EMAIL' => $borica_email,
            'COUNTRY' => $borica_country,
            'MERCH_GMT' => $borica_merch_gmt,
            'ADDENDUM' => $borica_addendum,
            'AD.CUST_BOR_ORDER_ID' => $borica_ad_cust_bor_order_id,
            'TIMESTAMP' => $borica_timestamp,
            'NONCE' => $borica_nonce,
            'LANG' => $borica_lang,
            'M_INFO' => base64_encode($borica_m_info_str),
            'P_SIGN' => isset($borica_p_sign['pSign']) ? $borica_p_sign['pSign'] : '',
        ];

        return $result;
    }

    /**
     * Transliterates Cyrillic characters to their Latin equivalents.
     *
     * This method takes a string containing Cyrillic characters and converts it to a Latin alphabet
     * representation. This is often necessary for compatibility with systems that do not support
     * non-Latin characters, such as certain payment gateways or databases.
     *
     * Key functionality:
     * - **Character Mapping:** The method uses a predefined mapping array `$translit` that pairs Cyrillic
     *   characters with their Latin equivalents. Both uppercase and lowercase letters are included.
     * - **String Replacement:** The `strtr()` function is used to replace each Cyrillic character in the input
     *   string with its corresponding Latin character, as defined in the `$translit` array.
     * - **Supports Bulgarian Cyrillic:** The transliteration covers Bulgarian Cyrillic characters, which are
     *   commonly used in the context of Bulgarian names and addresses.
     *
     * Usage:
     * - This method is typically used when preparing data for systems that require Latin characters, ensuring
     *   that names, addresses, and other text fields are correctly formatted.
     *
     * Example:
     * - Input: `"Примерен текст"`
     * - Output: `"Primeren tekst"`
     *
     * @param string $input The string containing Cyrillic characters to be transliterated.
     *
     * @return string The transliterated string with Latin characters.
     */
    public function transliterate(string $input): string
    {
        $translit = array(
            'а' => 'a',
            'б' => 'b',
            'в' => 'v',
            'г' => 'g',
            'д' => 'd',
            'е' => 'e',
            'ж' => 'zh',
            'з' => 'z',
            'и' => 'i',
            'й' => 'y',
            'к' => 'k',
            'л' => 'l',
            'м' => 'm',
            'н' => 'n',
            'о' => 'o',
            'п' => 'p',
            'р' => 'r',
            'с' => 's',
            'т' => 't',
            'у' => 'u',
            'ф' => 'f',
            'х' => 'h',
            'ц' => 'ts',
            'ч' => 'ch',
            'ш' => 'sh',
            'щ' => 'sht',
            'ъ' => 'a',
            'ь' => 'y',
            'ю' => 'yu',
            'я' => 'ya',
            'А' => 'A',
            'Б' => 'B',
            'В' => 'V',
            'Г' => 'G',
            'Д' => 'D',
            'Е' => 'E',
            'Ж' => 'Zh',
            'З' => 'Z',
            'И' => 'I',
            'Й' => 'Y',
            'К' => 'K',
            'Л' => 'L',
            'М' => 'M',
            'Н' => 'N',
            'О' => 'O',
            'П' => 'P',
            'Р' => 'R',
            'С' => 'S',
            'Т' => 'T',
            'У' => 'U',
            'Ф' => 'F',
            'Х' => 'H',
            'Ц' => 'Ts',
            'Ч' => 'Ch',
            'Ш' => 'Sh',
            'Щ' => 'Sht',
            'Ъ' => 'A',
            'Ь' => 'Y',
            'Ю' => 'Yu',
            'Я' => 'Ya',
        );
        return strtr($input, $translit);
    }
}
