<?php

require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaLogger.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaBoricaHelper.php';
require_once _PS_MODULE_DIR_ . 'mtborica/classes/MtboricaRecurring.php';
/**
 * @File: mtborica.php
 * @Author: BORICA AD
 * @Author e-mail: info@borica.bg
 * @Publisher: BORICA AD
 * @Publisher e-mail: info@borica.bg
 * @Owner: BORICA AD
 * @Version: 1.0.0
 */
if (!defined('_PS_VERSION_')) {
    exit;
}

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;
use OrderState;
use Language;

/**
 * Class Mtborica
 *
 * PrestaShop payment module for BORICA card payments with optional
 * recurring plan management. Provides configuration UI, DB install,
 * and hook integrations. The admin UI is rendered via
 * `views/templates/admin/configure.tpl` and powered by AJAX
 * controllers under `controllers/front`.
 *
 * @extends PaymentModule
 */
class Mtborica extends PaymentModule
{
    /**
     * @var array Admin controllers
     */
    public $controllers = ['AdminMtboricaOrder'];

    // Define Borica test environment URL.
    public const BORICA_TEST_URL = 'https://3dsgate-dev.borica.bg/cgi-bin/cgi_link';
    // Define Borica production environment URL.
    public const BORICA_PRODUCTION_URL = 'https://3dsgate.borica.bg/cgi-bin/cgi_link';
    // Define time in hours to check payment status.
    public const BORICA_CHECK_PAYMENT_STATUS_TIME = 24;
    // Define time in minutes to drop payment if not completed.
    public const BORICA_DROP_PAYMENT_TIME = 720;
    // Define transaction type code for checking payment status.
    public const BORICA_TRTYPE_CHECK_STATUS = 90;
    // Define transaction type code for payment authorization.
    public const BORICA_TRTYPE_AUTHORIZATION = 1;
    // Define transaction type code for dropping the payment status.
    public const BORICA_TRTYPE_DROP_STATUS = 24;
    // Define transaction type code for dropping the payment status.
    public const BORICA_TRTYPE_RECURRING_DROP_STATUS = 179;
    // Define default country code (BG for Bulgaria).
    public const BORICA_COUNTRY = 'BG';
    // Define default language code (BG for Bulgarian).
    public const BORICA_LANG = 'BG';
    // Define additional data fields (AD, TD) to include in transactions.
    public const BORICA_ADDENDUM = 'AD,TD';

    /**
     * Module constructor: sets identity, metadata and default texts.
     */
    public function __construct()
    {
        $this->name = 'mtborica';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.0';
        $this->author = 'BORICA AD';
        $this->need_instance = 0;
        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => '1.7.999.999'];
        $this->bootstrap = true;
        parent::__construct();
        $this->displayName = $this->l('Payment by Credit/Debit Card');
        $this->description = $this->l('Payment by Credit/Debit Card');
        $this->confirmUninstall = $this->l('Are you sure you want to uninstall this module?');
        if (!Configuration::get('MTBORICA_NAME'))
            $this->warning = $this->l('No name provided');
    }

    /**
     * Installs the module: registers hooks, seeds configuration values,
     * and creates required DB tables.
     *
     * @return bool True on success, false otherwise
     */
    public function install()
    {
        if (Shop::isFeatureActive()) {
            Shop::setContext(Shop::CONTEXT_ALL);
        }
        return parent::install() &&
            $this->registerHook('ActionFrontControllerSetMedia') &&
            $this->registerHook('paymentOptions') &&
            $this->registerHook('paymentReturn') &&
            $this->registerHook('displayAdminOrderTabLink') &&
            $this->registerHook('displayAdminOrderTabContent') &&
            $this->registerHook('displayAdminProductsExtra') &&
            $this->registerHook('displayProductButtons') &&
            $this->registerHook('actionProductUpdate') &&
            $this->registerHook('actionCartSave') &&
            $this->registerHook('displayProductPriceBlock') &&
            $this->registerHook('displayOrderDetail') &&
            Configuration::updateValue('MTBORICA_NAME', 'BORICA') &&
            Configuration::updateValue('mtborica_status', 0) &&
            Configuration::updateValue('mtborica_testmode', 1) &&
            Configuration::updateValue('mtborica_debug', 0) &&
            Configuration::updateValue('mtborica_mname', '') &&
            Configuration::updateValue('mtborica_unsuccess_message', 'Плащането е неуспешно') &&
            Configuration::updateValue('mtborica_success_message', 'Плащането е успешно') &&
            Configuration::updateValue('mtborica_email', '') &&
            Configuration::updateValue('mtborica_mid_bgn', '') &&
            Configuration::updateValue('mtborica_tid_bgn', '') &&
            Configuration::updateValue('mtborica_test_key_bgn', '') &&
            Configuration::updateValue('mtborica_test_password_bgn', '') &&
            Configuration::updateValue('mtborica_production_key_bgn', '') &&
            Configuration::updateValue('mtborica_production_password_bgn', '') &&
            Configuration::updateValue('mtborica_mid_eur', '') &&
            Configuration::updateValue('mtborica_tid_eur', '') &&
            Configuration::updateValue('mtborica_test_key_eur', '') &&
            Configuration::updateValue('mtborica_test_password_eur', '') &&
            Configuration::updateValue('mtborica_production_key_eur', '') &&
            Configuration::updateValue('mtborica_production_password_eur', '') &&
            Configuration::updateValue('mtborica_recurring', 0) &&
            Configuration::updateValue('mtborica_recurring_cancel_button', 0) &&
            Configuration::updateValue('mtborica_order_status_id', Configuration::get('PS_OS_PAYMENT')) &&
            Configuration::updateValue('mtborica_order_status_not_id', Configuration::get('PS_OS_CANCELED')) &&
            Configuration::updateValue('mtborica_payment_lang', 'BG') &&
            $this->installDb() &&
            $this->installOrderStates();
    }

    /**
     * Uninstalls the module: removes configuration values and drops
     * DB tables via uninstallDb().
     *
     * @return bool True on success, false otherwise
     */
    public function uninstall()
    {
        if (
            !parent::uninstall() ||
            !Configuration::deleteByName('MTBORICA_NAME') ||
            !Configuration::deleteByName('mtborica_status') ||
            !Configuration::deleteByName('mtborica_testmode') ||
            !Configuration::deleteByName('mtborica_debug') ||
            !Configuration::deleteByName('mtborica_mname') ||
            !Configuration::deleteByName('mtborica_unsuccess_message') ||
            !Configuration::deleteByName('mtborica_success_message') ||
            !Configuration::deleteByName('mtborica_email') ||
            !Configuration::deleteByName('mtborica_mid_bgn') ||
            !Configuration::deleteByName('mtborica_tid_bgn') ||
            !Configuration::deleteByName('mtborica_test_key_bgn') ||
            !Configuration::deleteByName('mtborica_test_password_bgn') ||
            !Configuration::deleteByName('mtborica_production_key_bgn') ||
            !Configuration::deleteByName('mtborica_production_password_bgn') ||
            !Configuration::deleteByName('mtborica_mid_eur') ||
            !Configuration::deleteByName('mtborica_tid_eur') ||
            !Configuration::deleteByName('mtborica_test_key_eur') ||
            !Configuration::deleteByName('mtborica_test_password_eur') ||
            !Configuration::deleteByName('mtborica_production_key_eur') ||
            !Configuration::deleteByName('mtborica_production_password_eur') ||
            !Configuration::deleteByName('mtborica_recurring') ||
            !Configuration::deleteByName('mtborica_recurring_cancel_button') ||
            !Configuration::deleteByName('mtborica_order_status_id') ||
            !Configuration::deleteByName('mtborica_order_status_not_id') ||
            !Configuration::deleteByName('mtborica_payment_lang') ||
            !Configuration::deleteByName('mtborica_order_status_recurring') ||
            !Configuration::deleteByName('mtborica_order_status_recurring_cancelled') ||
            !$this->uninstallOrderStates() ||
            !$this->uninstallDb()
        ) {
            return false;
        }
        return true;
    }

    /**
     * Handles module configuration form submission and renders the admin UI.
     * Performs server-side validation for all inputs and persists settings.
     *
     * @return string Rendered HTML of the configuration page
     */
    public function getContent()
    {
        $output = null;
        $mtborica_error = false;
        if (Tools::isSubmit('submit' . $this->name)) {
            $mtborica_status = (int) Tools::getValue('mtborica_status');
            $mtborica_testmode = (int) Tools::getValue('mtborica_testmode');
            $mtborica_debug = (int) Tools::getValue('mtborica_debug');
            $mtborica_mname = (string) Tools::getValue('mtborica_mname');
            if (null === $mtborica_mname || '' === $mtborica_mname || !Validate::isGenericName($mtborica_mname)) {
                $output .= $this->displayError('Wrong field value "Merchant Name"');
                $mtborica_error = true;
            }
            $mtborica_unsuccess_message = (string) Tools::getValue('mtborica_unsuccess_message');
            if (null === $mtborica_unsuccess_message || '' === $mtborica_unsuccess_message || !Validate::isGenericName($mtborica_unsuccess_message)) {
                $output .= $this->displayError('Wrong field value "Unsuccess Message"');
                $mtborica_error = true;
            }
            $mtborica_success_message = (string) Tools::getValue('mtborica_success_message');
            if (null === $mtborica_success_message || '' === $mtborica_success_message || !Validate::isGenericName($mtborica_success_message)) {
                $output .= $this->displayError('Wrong field value "Success Message"');
                $mtborica_error = true;
            }
            $mtborica_email = (string) Tools::getValue('mtborica_email');
            if (null === $mtborica_email || '' === $mtborica_email || !Validate::isEmail($mtborica_email)) {
                $output .= $this->displayError('Wrong field value "Email"');
                $mtborica_error = true;
            }
            $mtborica_mid_bgn = (string) Tools::getValue('mtborica_mid_bgn');
            $mtborica_tid_bgn = (string) Tools::getValue('mtborica_tid_bgn');
            $mtborica_test_key_bgn = (string) Tools::getValue('mtborica_test_key_bgn');
            if (!empty($mtborica_test_key_bgn) && !$this->validateSslPrivateKey($mtborica_test_key_bgn)) {
                $output .= $this->displayError('Wrong field value "Test Key BGN" - invalid SSL private key');
                $mtborica_error = true;
            }
            $mtborica_test_password_bgn = (string) Tools::getValue('mtborica_test_password_bgn');
            $mtborica_production_key_bgn = (string) Tools::getValue('mtborica_production_key_bgn');
            if (!empty($mtborica_production_key_bgn) && !$this->validateSslPrivateKey($mtborica_production_key_bgn)) {
                $output .= $this->displayError('Wrong field value "Production Key BGN" - invalid SSL private key');
                $mtborica_error = true;
            }
            $mtborica_production_password_bgn = (string) Tools::getValue('mtborica_production_password_bgn');
            if (!empty($mtborica_production_key_bgn) && empty($mtborica_production_password_bgn)) {
                $output .= $this->displayError('Production Key Password BGN is required when Production Key BGN is provided');
                $mtborica_error = true;
            }
            $mtborica_mid_eur = (string) Tools::getValue('mtborica_mid_eur');
            $mtborica_tid_eur = (string) Tools::getValue('mtborica_tid_eur');
            $mtborica_test_key_eur = (string) Tools::getValue('mtborica_test_key_eur');
            if (!empty($mtborica_test_key_eur) && !$this->validateSslPrivateKey($mtborica_test_key_eur)) {
                $output .= $this->displayError('Wrong field value "Test Key EUR" - invalid SSL private key');
                $mtborica_error = true;
            }
            $mtborica_test_password_eur = (string) Tools::getValue('mtborica_test_password_eur');
            $mtborica_production_key_eur = (string) Tools::getValue('mtborica_production_key_eur');
            if (!empty($mtborica_production_key_eur) && !$this->validateSslPrivateKey($mtborica_production_key_eur)) {
                $output .= $this->displayError('Wrong field value "Production Key EUR" - invalid SSL private key');
                $mtborica_error = true;
            }
            $mtborica_production_password_eur = (string) Tools::getValue('mtborica_production_password_eur');
            if (!empty($mtborica_production_key_eur) && empty($mtborica_production_password_eur)) {
                $output .= $this->displayError('Production Key Password EUR is required when Production Key EUR is provided');
                $mtborica_error = true;
            }
            $mtborica_recurring = (int) Tools::getValue('mtborica_recurring');
            $mtborica_recurring_cancel_button = (int) Tools::getValue('mtborica_recurring_cancel_button');
            $mtborica_order_status_id = (int) Tools::getValue('mtborica_order_status_id');
            if ($mtborica_order_status_id <= 0) {
                $output .= $this->displayError('Wrong field value "Order status after successful payment"');
                $mtborica_error = true;
            }
            $mtborica_order_status_not_id = (int) Tools::getValue('mtborica_order_status_not_id');
            if ($mtborica_order_status_not_id <= 0) {
                $output .= $this->displayError('Order status after unsuccessful payment"');
                $mtborica_error = true;
            }
            $mtborica_payment_lang = (string) Tools::getValue('mtborica_payment_lang');

            if (!$mtborica_error) {
                Configuration::updateValue('mtborica_status', $mtborica_status);
                Configuration::updateValue('mtborica_testmode', $mtborica_testmode);
                Configuration::updateValue('mtborica_debug', $mtborica_debug);
                Configuration::updateValue('mtborica_mname', $mtborica_mname);
                Configuration::updateValue('mtborica_unsuccess_message', $mtborica_unsuccess_message);
                Configuration::updateValue('mtborica_success_message', $mtborica_success_message);
                Configuration::updateValue('mtborica_email', $mtborica_email);
                Configuration::updateValue('mtborica_mid_bgn', $mtborica_mid_bgn);
                Configuration::updateValue('mtborica_tid_bgn', $mtborica_tid_bgn);
                Configuration::updateValue('mtborica_test_key_bgn', $mtborica_test_key_bgn);
                Configuration::updateValue('mtborica_test_password_bgn', $mtborica_test_password_bgn);
                Configuration::updateValue('mtborica_production_key_bgn', $mtborica_production_key_bgn);
                Configuration::updateValue('mtborica_production_password_bgn', $mtborica_production_password_bgn);
                Configuration::updateValue('mtborica_mid_eur', $mtborica_mid_eur);
                Configuration::updateValue('mtborica_tid_eur', $mtborica_tid_eur);
                Configuration::updateValue('mtborica_test_key_eur', $mtborica_test_key_eur);
                Configuration::updateValue('mtborica_test_password_eur', $mtborica_test_password_eur);
                Configuration::updateValue('mtborica_production_key_eur', $mtborica_production_key_eur);
                Configuration::updateValue('mtborica_production_password_eur', $mtborica_production_password_eur);
                Configuration::updateValue('mtborica_recurring', $mtborica_recurring);
                Configuration::updateValue('mtborica_recurring_cancel_button', $mtborica_recurring_cancel_button);
                Configuration::updateValue('mtborica_payment_lang', $mtborica_payment_lang);
                Configuration::updateValue('mtborica_order_status_id', $mtborica_order_status_id);
                Configuration::updateValue('mtborica_order_status_not_id', $mtborica_order_status_not_id);
                $output .= $this->displayConfirmation('The changes have been successfully saved');
            }
        }

        return $output . $this->displayForm();
    }

    /**
     * Assigns Smarty variables and renders the configuration template.
     *
     * @return string HTML of the form
     */
    private function displayForm()
    {
        // Loads the order states
        $orderStates = OrderState::getOrderStates((int) $this->context->language->id);

        // Loads the values from the configuration
        $this->context->smarty->assign(array(
            'mtborica_status' => Configuration::get('mtborica_status'),
            'mtborica_testmode' => Configuration::get('mtborica_testmode'),
            'mtborica_debug' => Configuration::get('mtborica_debug'),
            'mtborica_mname' => Configuration::get('mtborica_mname'),
            'mtborica_unsuccess_message' => Configuration::get('mtborica_unsuccess_message'),
            'mtborica_success_message' => Configuration::get('mtborica_success_message'),
            'mtborica_email' => Configuration::get('mtborica_email'),
            'mtborica_mid_bgn' => Configuration::get('mtborica_mid_bgn'),
            'mtborica_tid_bgn' => Configuration::get('mtborica_tid_bgn'),
            'mtborica_test_key_bgn' => Configuration::get('mtborica_test_key_bgn'),
            'mtborica_test_password_bgn' => Configuration::get('mtborica_test_password_bgn'),
            'mtborica_production_key_bgn' => Configuration::get('mtborica_production_key_bgn'),
            'mtborica_production_password_bgn' => Configuration::get('mtborica_production_password_bgn'),
            'mtborica_mid_eur' => Configuration::get('mtborica_mid_eur'),
            'mtborica_tid_eur' => Configuration::get('mtborica_tid_eur'),
            'mtborica_test_key_eur' => Configuration::get('mtborica_test_key_eur'),
            'mtborica_test_password_eur' => Configuration::get('mtborica_test_password_eur'),
            'mtborica_production_key_eur' => Configuration::get('mtborica_production_key_eur'),
            'mtborica_production_password_eur' => Configuration::get('mtborica_production_password_eur'),
            'mtborica_recurring' => Configuration::get('mtborica_recurring'),
            'mtborica_recurring_cancel_button' => Configuration::get('mtborica_recurring_cancel_button'),
            'mtborica_payment_lang' => Configuration::get('mtborica_payment_lang'),
            'mtborica_order_status_id' => Configuration::get('mtborica_order_status_id'),
            'mtborica_order_status_not_id' => Configuration::get('mtborica_order_status_not_id'),
            'mtborica_recurrings' => $this->getRecurringPayments(),
            'order_states' => $orderStates,
            'mtborica_translations' => array(
                'recurring_plan_saved_success' => $this->l('Recurring plan has been saved successfully.'),
                'recurring_plan_updated_success' => $this->l('Recurring plan updated successfully.'),
                'recurring_plan_save_failed' => $this->l('Failed to save recurring plan.'),
            ),
            'mtborica_test_keys_bgn_url' => $this->context->link->getModuleLink($this->name, 'testKeysBgn', array()),
            'mtborica_production_keys_bgn_url' => $this->context->link->getModuleLink($this->name, 'productionKeysBgn', array()),
            'mtborica_test_keys_eur_url' => $this->context->link->getModuleLink($this->name, 'testKeysEur', array()),
            'mtborica_production_keys_eur_url' => $this->context->link->getModuleLink($this->name, 'productionKeysEur', array()),
            'mtborica_modal_save_url' => $this->context->link->getModuleLink($this->name, 'createReccuring', array()),
            'mtborica_toggle_status_url' => $this->context->link->getModuleLink($this->name, 'toggleRecurringStatus', array()),
            'mtborica_get_recurring_url' => $this->context->link->getModuleLink($this->name, 'getRecurring', array()),
            'mtborica_update_recurring_url' => $this->context->link->getModuleLink($this->name, 'updateRecurring', array()),
            'mtborica_delete_recurring_url' => $this->context->link->getModuleLink($this->name, 'deleteRecurring', array()),
            'mtborica_back_url' => $this->context->shop->getBaseURL(true) . 'index.php?fc=module&module=' . $this->name . '&controller=back',
        ));

        return $this->display(__FILE__, 'views/templates/admin/configure.tpl');
    }

    /**
     * Returns list of recurring payments for the admin UI table.
     *
     * @return array Array of recurring rows
     */
    private function getRecurringPayments()
    {
        // Include the model class
        require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';

        return MtboricaRecurring::getAll();
    }

    /**
     * Hook: ActionFrontControllerSetMedia.
     * Reserved for enqueueing assets on the front-office.
     *
     * @param array $params Hook parameters
     * @return array|null Not used currently
     */
    public function hookActionFrontControllerSetMedia($params)
    {
        if ('order' === $this->context->controller->php_self) {
            $this->context->controller->registerStylesheet(
                'module-mtborica-checkout-css',
                'modules/' . $this->name . '/css/mtborica_checkout.css',
                [
                    'media' => 'all',
                    'priority' => 200,
                    'version' => filemtime(_PS_MODULE_DIR_ . $this->name . '/css/mtborica_checkout.css'),
                ]
            );
            $this->context->controller->registerJavascript(
                'module-mtborica-checkout-js',
                'modules/' . $this->name . '/js/mtborica_checkout.js',
                [
                    'priority' => 200,
                    'attribute' => 'defer',
                    'version' => filemtime(_PS_MODULE_DIR_ . $this->name . '/js/mtborica_checkout.js'),
                ]
            );
        }
    }

    /**
     * Hook: paymentOptions.
     * Returns available payment options on checkout. Currently empty
     * (module is configured for back-office only operations here).
     *
     * @param array $params Hook parameters
     * @return array PaymentOption[] list (empty for now)
     */
    public function hookPaymentOptions($params)
    {
        try {
            if (empty($params['cart'])) {
                return [];
            }

            $cart = $params['cart'];

            if ($cart->isVirtualCart()) {
                return [];
            }

            $mtborica_status = (int) Configuration::get('mtborica_status');
            if ($mtborica_status == 0) {
                return [];
            }

            $mtborica_currency = $this->context->currency->iso_code;
            if ($mtborica_currency != 'BGN' && $mtborica_currency != 'EUR') {
                return [];
            }
            $mtborica_currency_symbol = $this->context->currency->sign;

            // Perform cleanup to ensure cart plan records are in sync before validation
            if (self::isRecurringEnabled()) {
                self::cleanupOrphanedCartPlans((int) $cart->id);
            }

            // Check for recurring payments
            $recurring_plan_id = 0;
            $is_recurring_payment = false;
            $recurring_plan = null;

            if (self::isRecurringEnabled()) {
                // Check if cart has mixed products (recurring + standard) - don't show payment option
                if (self::cartHasMixedProducts($cart)) {
                    return []; // Don't show payment option if cart has mixed products
                }

                // Check if cart has only recurring products with the same plan
                $recurring_plan_id = self::cartHasSingleRecurringPlan($cart);
                if ($recurring_plan_id > 0) {
                    $is_recurring_payment = true;
                    // Load plan information
                    require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';
                    $recurring_plan = MtboricaRecurring::getById($recurring_plan_id);
                }
            }

            $mtborica_testmode = (int) Configuration::get('mtborica_testmode');

            // Prepare template variables
            $template_vars = [
                'borica_testmode' => Configuration::get('mtborica_testmode'),
                'mtborica_cards' => __PS_BASE_URI__ . 'modules/mtborica/views/img/borica_cards.png',
                'mtborica_display_name' => !empty($this->displayName) ? $this->displayName : 'BORICA',
                'mtborica_description' => !empty($this->description) ? $this->description : 'Payment by Credit/Debit Card',
                'is_recurring_payment' => $is_recurring_payment,
                'recurring_plan_id' => $recurring_plan_id,
                'is_total_amount_mismatch' => false,
            ];

            // Add recurring plan information if available
            if ($is_recurring_payment && $recurring_plan && $recurring_plan->id) {
                $borica_amount_arr = MtboricaBoricaHelper::calculateFirstPaymentAmount(
                    (float) $cart->getOrderTotal(),
                    $recurring_plan->recur_duration,
                    $recurring_plan->recur_duration_unit,
                    $recurring_plan->recur_freq,
                    $recurring_plan->recur_freq_unit
                );
                $borica_payments = is_array($borica_amount_arr) ? (float) $borica_amount_arr['payments'] : 1;
                $borica_amount = is_array($borica_amount_arr) ? (float) $borica_amount_arr['amount'] : 0.00;
                $borica_total_amount = round($borica_payments * $borica_amount, 2);

                $template_vars['borica_payments'] = $borica_payments;
                $template_vars['borica_amount'] = $borica_amount;
                $template_vars['mtborica_currency_symbol'] = $mtborica_currency_symbol;
                $template_vars['borica_total_amount'] = $borica_total_amount;
                if ($borica_total_amount !== round((float) $cart->getOrderTotal(), 2)) {
                    $template_vars['is_total_amount_mismatch'] = true;
                }
            }

            $this->context->smarty->assign($template_vars);

            // Set payment title based on payment type
            if ($is_recurring_payment && $recurring_plan && $recurring_plan->id) {
                $mtborica_title = $this->l('Recurring Payment by Credit/Debit Card');
                if ($mtborica_testmode == 1) {
                    $mtborica_title .= ' (' . $this->l('TEST MODE') . ')';
                }
            } else {
                $mtborica_title = $this->l('Payment by Credit/Debit Card');
                if ($mtborica_testmode == 1) {
                    $mtborica_title .= ' (' . $this->l('TEST MODE') . ')';
                }
            }

            $payment_options = [];

            $newOption_MTBORICA = new PaymentOption();
            $newOption_MTBORICA->setModuleName($this->name);
            $newOption_MTBORICA->setCallToActionText($mtborica_title);
            $newOption_MTBORICA->setAction(
                $this->context->link->getModuleLink(
                    $this->name,
                    'validation',
                    [],
                    true
                )
            );

            // Use different template for recurring payments
            $template_name = $is_recurring_payment
                ? 'module:mtborica/views/templates/hook/mtborica_checkout_recurring.tpl'
                : 'module:mtborica/views/templates/hook/mtborica_checkout.tpl';

            $newOption_MTBORICA->setAdditionalInformation($this->fetch($template_name));

            // Add hidden fields to form so they are submitted with the payment
            $hidden_inputs = [
                [
                    'type' => 'hidden',
                    'name' => 'browserScreenHeight',
                    'value' => '0', // Will be updated by JavaScript before submit
                ],
                [
                    'type' => 'hidden',
                    'name' => 'browserScreenWidth',
                    'value' => '0', // Will be updated by JavaScript before submit
                ],
            ];

            // Add recurring plan ID if this is a recurring payment
            if ($is_recurring_payment && $recurring_plan_id > 0) {
                $hidden_inputs[] = [
                    'type' => 'hidden',
                    'name' => 'mtborica_recurring_plan_id',
                    'value' => (string) $recurring_plan_id,
                ];
            }

            $newOption_MTBORICA->setInputs($hidden_inputs);
            $payment_options[] = $newOption_MTBORICA;

            return $payment_options;
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Creates required database tables for recurring plans and orders log.
     *
     * @return bool True on success, false otherwise
     */
    private function installDb()
    {
        $sql_recurring = "CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "mtborica_recurring` (
            `id` INT(11) NOT NULL AUTO_INCREMENT,
            `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `name` VARCHAR(128) NOT NULL,
            `status` TINYINT UNSIGNED NOT NULL DEFAULT 1,
            `recur_duration` varchar(4) NOT NULL,
            `recur_duration_unit` ENUM('D', 'M', 'Q', 'H', 'Y') NOT NULL DEFAULT 'D',
            `recur_freq` varchar(4) NOT NULL,
            `recur_freq_unit` ENUM('D', 'M', 'Q', 'H', 'Y') NOT NULL DEFAULT 'D',
            `recur_mday_payment` varchar(2) NOT NULL,
            KEY idx_status (status),
            PRIMARY KEY (`id`)
        ) ENGINE=" . _MYSQL_ENGINE_ . " DEFAULT CHARSET=utf8;";

        $sql_orders = "CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "mtborica_orders` (
            `id` INT(11) NOT NULL AUTO_INCREMENT,
            `action` varchar(20) NOT NULL,
            `rc` varchar(20) NOT NULL,
            `created_at` DATETIME NOT NULL,
            `increment_id` varchar(50) NULL,
            `status` varchar(255) NOT NULL,
            `rrn` varchar(20) NOT NULL,
            `int_ref` varchar(20) NOT NULL,
            `approval` varchar(6) NOT NULL,
            `merch_gmt` varchar(3) NOT NULL,
            `request_cancel` varchar(20) NOT NULL,
            `cancel_amount` varchar(20) NOT NULL,
            `nonce` varchar(32) NOT NULL,
            `merch_rn_id` varchar(16) NOT NULL,
            `recur_id` varchar(9) NOT NULL,
            `amount` varchar(20) NOT NULL,
            PRIMARY KEY (`id`)
        ) ENGINE=" . _MYSQL_ENGINE_ . " DEFAULT CHARSET=utf8;";

        $sql_payloads = "CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "mtborica_payloads` (
            `id` INT UNSIGNED NOT NULL AUTO_INCREMENT,
            `token` VARCHAR(64) NOT NULL,
            `payload_json` LONGTEXT NOT NULL,
            `date_add` DATETIME NOT NULL,
            `date_expire` DATETIME DEFAULT NULL,
            PRIMARY KEY (`id`),
            UNIQUE KEY `ux_token` (`token`)
        ) ENGINE=" . _MYSQL_ENGINE_ . " DEFAULT CHARSET=utf8;";

        $sql_cart_products = "CREATE TABLE IF NOT EXISTS `" . _DB_PREFIX_ . "mtborica_cart_products` (
            `id` INT(11) NOT NULL AUTO_INCREMENT,
            `id_cart` INT(10) UNSIGNED NOT NULL,
            `id_product` INT(10) UNSIGNED NOT NULL,
            `id_product_attribute` INT(10) UNSIGNED NOT NULL DEFAULT 0,
            `id_recurring_plan` INT(11) NOT NULL DEFAULT 0,
            `date_add` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `date_upd` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `ux_cart_product` (`id_cart`, `id_product`, `id_product_attribute`),
            KEY `idx_cart` (`id_cart`),
            KEY `idx_product` (`id_product`),
            KEY `idx_recurring_plan` (`id_recurring_plan`)
        ) ENGINE=" . _MYSQL_ENGINE_ . " DEFAULT CHARSET=utf8;";

        $db = Db::getInstance();
        $okRecurring = (bool) $db->execute($sql_recurring);
        $okOrders = (bool) $db->execute($sql_orders);
        $okPayloads = (bool) $db->execute($sql_payloads);
        $okCartProducts = (bool) $db->execute($sql_cart_products);

        return ($okRecurring && $okOrders && $okPayloads && $okCartProducts);
    }

    /**
     * Drops module database tables on uninstall.
     *
     * @return bool True on success, false otherwise
     */
    private function uninstallDb()
    {
        $sql_recurring = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mtborica_recurring`;';
        $sql_orders = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mtborica_orders`;';
        $sql_payloads = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mtborica_payloads`;';
        $sql_cart_products = 'DROP TABLE IF EXISTS `' . _DB_PREFIX_ . 'mtborica_cart_products`;';

        $db = Db::getInstance();
        $okRecurring = (bool) $db->execute($sql_recurring);
        $okOrders = (bool) $db->execute($sql_orders);
        $okPayloads = (bool) $db->execute($sql_payloads);
        $okCartProducts = (bool) $db->execute($sql_cart_products);

        return ($okRecurring && $okOrders && $okPayloads && $okCartProducts);
    }

    public function hookPaymentReturn($params)
    {
        $token = Tools::getValue('borica_token');
        $borica_payment_data = null;
        if ($token) {
            $query = new DbQuery();
            $query->select('payload_json')
                ->from('mtborica_payloads')
                ->where('token = \'' . pSQL($token) . '\'');
            $result = Db::getInstance()->getRow($query);
            if ($result !== false) {
                $borica_payment_data = $result['payload_json'];
                Db::getInstance()->delete('mtborica_payloads', 'token = \'' . pSQL($token) . '\'', 1);
            }
        }

        if (!$this->active) {
            return null;
        }

        // Always render when token payload exists, regardless of order state
        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        $borica_url = ($borica_testmode == 1)
            ? self::BORICA_TEST_URL
            : self::BORICA_PRODUCTION_URL;

        if (!$borica_payment_data) {
            return null;
        }

        $borica_payment_data = json_decode($borica_payment_data, true);

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logPaymentSent($borica_payment_data);
        }

        $this->context->smarty->assign(array(
            'borica_url' => $borica_url,
            'borica_payment_data' => $borica_payment_data,
        ));

        // Use explicit path to the front template
        return $this->fetch('module:mtborica/views/templates/front/payment_return.tpl');
    }

    /**
     * Validates whether provided string looks like a PEM-encoded private key.
     * Checks header/footer markers, base64 body and minimal length.
     *
     * @param string $key PEM-formatted private key content
     * @return bool True if the key format appears valid, false otherwise
     */
    private function validateSslPrivateKey($key)
    {
        // Checks if the key is empty
        if (empty($key) || trim($key) === '') {
            return false;
        }

        // Removes the empty lines at the beginning and end
        $key = trim($key);

        // Checks if the key starts with the correct header
        $validHeaders = [
            '-----BEGIN PRIVATE KEY-----',
            '-----BEGIN RSA PRIVATE KEY-----',
            '-----BEGIN EC PRIVATE KEY-----',
            '-----BEGIN DSA PRIVATE KEY-----',
            '-----BEGIN ENCRYPTED PRIVATE KEY-----'
        ];

        $hasValidHeader = false;
        foreach ($validHeaders as $header) {
            if (strpos($key, $header) === 0) {
                $hasValidHeader = true;
                break;
            }
        }

        if (!$hasValidHeader) {
            return false;
        }

        // Checks if the key ends with the correct footer
        $validFooters = [
            '-----END PRIVATE KEY-----',
            '-----END RSA PRIVATE KEY-----',
            '-----END EC PRIVATE KEY-----',
            '-----END DSA PRIVATE KEY-----',
            '-----END ENCRYPTED PRIVATE KEY-----'
        ];

        $hasValidFooter = false;
        foreach ($validFooters as $footer) {
            if (substr($key, -strlen($footer)) === $footer) {
                $hasValidFooter = true;
                break;
            }
        }

        if (!$hasValidFooter) {
            return false;
        }

        // Checks if the key contains only valid symbols (Base64 + new lines)
        $keyContent = str_replace(["\r\n", "\n", "\r"], '', $key);
        $keyContent = preg_replace('/^-----BEGIN .*?-----/', '', $keyContent);
        $keyContent = preg_replace('/-----END .*?-----$/', '', $keyContent);

        // Checks if the remaining content is a valid Base64
        if (!empty($keyContent) && !preg_match('/^[A-Za-z0-9+\/]*={0,2}$/', $keyContent)) {
            return false;
        }

        // Checks the minimum length of the key (usually SSL keys are at least 1000 characters)
        if (strlen($key) < 1000) {
            return false;
        }

        return true;
    }

    /**
     * Hook: displayAdminOrderTabLink
     * Adds a tab link in the order view page
     *
     * @param array $params Hook parameters (contains id_order)
     * @return string HTML for tab link
     */
    public function hookDisplayAdminOrderTabLink($params)
    {
        $id_order = (int) $params['id_order'];

        // Check if this order uses our payment method
        $order = new Order($id_order);
        if (!Validate::isLoadedObject($order) || $order->module !== $this->name) {
            return '';
        }

        $this->context->smarty->assign([
            'tab_label' => $this->l('BORICA - Payment by Credit/Debit Card'),
        ]);

        return $this->display(__FILE__, 'views/templates/admin/order_tab_link.tpl');
    }

    /**
     * Hook: displayAdminOrderTabContent
     * Adds tab content in the order view page
     *
     * @param array $params Hook parameters (contains id_order)
     * @return string HTML for tab content
     */
    public function hookDisplayAdminOrderTabContent($params)
    {
        $id_order = (int) $params['id_order'];

        // Check if this order uses our payment method
        $order = new Order($id_order);
        if (!Validate::isLoadedObject($order) || $order->module !== $this->name) {
            return '';
        }

        // Get payment data from database
        $paymentData = $this->getOrderPaymentData($id_order);
        if (!$paymentData) {
            return '';
        }

        // Get admin controller URL for actions
        $adminUrl = $this->context->link->getAdminLink('AdminMtboricaOrder', true, [], [
            'id_order' => $id_order,
        ]);

        // Generate admin token
        $token = Tools::getAdminTokenLite('AdminMtboricaOrder');
        // Use order total with taxes included from the Order model
        $borica_order_total = (float) $order->total_paid_tax_incl;

        // Resolve order currency symbol and ISO code via PrestaShop models
        $currency = new Currency((int) $order->id_currency);
        $currency_sign = !empty($currency->sign) ? $currency->sign : (property_exists($currency, 'symbol') && !empty($currency->symbol) ? $currency->symbol : '');

        $borica_reccuring_name = '';
        $recurreng_paln_cancelled = false;

        $borica_action = (int) $paymentData['action'];
        $borica_order_action_text = '';
        $borica_order_action_class = '';
        switch ($borica_action) {
            case 0:
                $borica_order_action_text = $this->l('Successfully completed transaction');
                $borica_order_action_class = 'text-success';
                break;
            case 1:
                $borica_order_action_text = $this->l('Duplicate transaction');
                $borica_order_action_class = 'text-danger';
                break;
            case 2:
                $borica_order_action_text = $this->l('Transaction declined');
                $borica_order_action_class = 'text-danger';
                break;
            case 3:
                $borica_order_action_text = $this->l('Error processing transaction');
                $borica_order_action_class = 'text-danger';
                break;
            case 7:
                $borica_order_action_text = $this->l('Duplicate transaction on failed authentication');
                $borica_order_action_class = 'text-danger';
                break;
            case 21:
                $borica_order_action_text = $this->l('Soft Decline');
                $borica_order_action_class = 'text-warning';
                break;
            case 999:
                $borica_order_action_text = $this->l('Transaction not completed');
                $borica_order_action_class = 'text-danger';
                break;
            default:
                $borica_order_action_text = $this->l('Transaction not completed');
                $borica_order_action_class = 'text-danger';
        }
        if (strlen($id_order) >= 6) {
            $borica_order = substr($id_order, -6);
            $borica_order_internal = $borica_order . $id_order;
        } else {
            $borica_order = str_pad($id_order, 6, '0', STR_PAD_LEFT);
            $borica_order_internal = $borica_order . $id_order;
        }

        $borica_payment_total = $borica_order_total;
        $recurring_row = MtboricaRecurring::getById($paymentData['recur_id']);
        if ($recurring_row) {
            $borica_reccuring_name = $recurring_row->name;
            $borica_payment_total = (float) $paymentData['amount'];
        }

        $borica_request_cancel_amount = (float) $paymentData['cancel_amount'];
        $borica_rest = $borica_payment_total - $borica_request_cancel_amount;
        $borica_request_cancel_text = '';
        $borica_request_cancel_class = 'text-muted';
        switch ($paymentData['request_cancel']) {
            case '00':
                $borica_request_cancel_class = 'text-success';
                if (0.0 === (float) $borica_rest) {
                    $borica_request_cancel_text =
                        $this->l('Successful cancellation of payment (or refund).') .
                        ' ' .
                        $this->l('Net amount paid by card ') .
                        ' ' .
                        number_format($borica_rest, 2, '.', '') . ' ' . $currency_sign;
                } else {
                    $borica_request_cancel_text =
                        $this->l('Amount successfully canceled ') .
                        ' ' .
                        number_format($borica_rest, 2, '.', '') . ' ' . $currency_sign .
                        ' ' .
                        $this->l('Net amount paid by card ') .
                        ' ' .
                        number_format($borica_request_cancel_amount, 2, '.', '') . ' ' . $currency_sign;
                }
                break;
            case '11':
                $borica_request_cancel_class = 'text-danger';
                $borica_request_cancel_text =
                    $this->l('Payment cancellation request sent. The request has been rejected.') .
                    ' ' .
                    $this->l('The request was to cancel the amount of: ') .
                    ' ' .
                    number_format($borica_rest, 2, '.', '') . '. ' .
                    $this->l('In the event of an unsuccessful cancellation, the merchant may contact the servicing financial institution.');
                break;
            case '999':
                $borica_request_cancel_class = 'text-muted';
                $borica_request_cancel_text = $this->l('No Request for Reversal Payment has been sent.');
                break;
        }

        $current_date_check = date('d.m.Y H:i:s', strtotime('-' . self::BORICA_CHECK_PAYMENT_STATUS_TIME . ' hours'));
        if (new DateTime($paymentData['created_at']) > new DateTime($current_date_check)) {
            $is_bcps = 1;
        } else {
            $is_bcps = 0;
        }

        $current_date_drop = date('d.m.Y H:i:s', strtotime('-' . self::BORICA_DROP_PAYMENT_TIME . ' hours'));
        if (
            (new DateTime($paymentData['created_at']) > new DateTime($current_date_drop)) &&
            '00' === $paymentData['rc'] &&
            0 === $borica_action &&
            '999' === $paymentData['request_cancel']
        ) {
            $is_bdp = 1;
        } else {
            $is_bdp = 0;
        }

        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        $borica_url = ($borica_testmode == 1)
            ? self::BORICA_TEST_URL
            : self::BORICA_PRODUCTION_URL;

        $borica_terminal = '';
        $borica_merchant = '';
        if ('BGN' === $currency->iso_code) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_bgn');
            $borica_merchant = (string) Configuration::get('mtborica_mid_bgn');
        }
        if ('EUR' === $currency->iso_code) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_eur');
            $borica_merchant = (string) Configuration::get('mtborica_mid_eur');
        }

        if ('999' === $paymentData['request_cancel']) {
            $borica_tran_trtype = self::BORICA_TRTYPE_AUTHORIZATION;
        } else {
            $borica_tran_trtype = self::BORICA_TRTYPE_DROP_STATUS;
        }

        $borica_mname = (string) Configuration::get('mtborica_mname');
        $borica_email = (string) Configuration::get('mtborica_email');
        $borica_lang = (string) Configuration::get('mtborica_payment_lang');
        $borica_timestamp = gmdate('YmdHis');

        $this->context->smarty->assign([
            'payment_data' => $paymentData,
            'admin_url' => $adminUrl,
            'token' => $token,
            'currency_sign' => $currency_sign,
            'borica_reccuring_name' => $borica_reccuring_name,
            'recurreng_paln_cancelled' => $recurreng_paln_cancelled,
            'borica_order_action_text' => $borica_order_action_text,
            'borica_order_action_class' => $borica_order_action_class,
            'borica_order_internal' => $borica_order_internal,
            'borica_request_cancel_text' => $borica_request_cancel_text,
            'borica_request_cancel_class' => $borica_request_cancel_class,
            'borica_url' => $borica_url,
            'borica_terminal' => $borica_terminal,
            'borica_merchant' => $borica_merchant,
            'borica_check_payment_trtype' => self::BORICA_TRTYPE_CHECK_STATUS,
            'borica_cancel_payment_trtype' => self::BORICA_TRTYPE_DROP_STATUS,
            'borica_order' => $borica_order,
            'borica_tran_trtype' => $borica_tran_trtype,
            'borica_currency' => $currency->iso_code,
            'base_url' => $this->context->link->getBaseLink(),
            'borica_mname' => $borica_mname,
            'borica_email' => $borica_email,
            'borica_country' => self::BORICA_COUNTRY,
            'borica_lang' => $borica_lang,
            'borica_addendum' => self::BORICA_ADDENDUM,
            'borica_timestamp' => $borica_timestamp,
            'borica_payment_total' => $borica_payment_total,
            'borica_order_total' => $borica_order_total,
            'is_bcps' => $is_bcps,
            'is_bdp' => $is_bdp,
            'order_status' => $this->getOrderStatusNameIfCancelled($order),
        ]);

        return $this->display(__FILE__, 'views/templates/admin/order_tab_content.tpl');
    }

    /**
     * Get order status name only if it's "Cancelled Recurring payments"
     *
     * @param Order $order Order object
     * @return string|null Status name if cancelled recurring, null otherwise
     */
    private function getOrderStatusNameIfCancelled($order)
    {
        $cancelled_status_id = (int) Configuration::get('mtborica_order_status_recurring_cancelled');
        $current_status_id = (int) $order->getCurrentState();

        if ($cancelled_status_id > 0 && $current_status_id === $cancelled_status_id) {
            $orderState = new OrderState($current_status_id);
            if (Validate::isLoadedObject($orderState)) {
                $id_lang = (int) $this->context->language->id;
                return isset($orderState->name[$id_lang]) ? $orderState->name[$id_lang] : null;
            }
        }

        return null;
    }

    /**
     * Get payment data for an order from mtborica_orders table
     *
     * @param int $id_order Order ID
     * @return array|null Payment data or null if not found
     */
    public function getOrderPaymentData($id_order)
    {
        $query = new DbQuery();
        $query->select('*')
            ->from('mtborica_orders')
            ->where('increment_id = \'' . pSQL($id_order) . '\'')
            ->orderBy('created_at DESC');

        $result = Db::getInstance()->getRow($query);

        return $result ? $result : null;
    }

    /**
     * Check if recurring payments feature is enabled
     *
     * @return bool True if recurring payments are enabled
     */
    public static function isRecurringEnabled()
    {
        return (int) Configuration::get('mtborica_recurring') === 1;
    }

    /**
     * Hook: Display extra fields in product form
     * Adds recurring plan selection field to product edit page
     * Shows in the "Modules" tab in product edit page
     * Only shows if recurring feature is enabled
     *
     * @param array $params Hook parameters
     * @return string HTML content (empty if recurring is disabled)
     */
    public function hookDisplayAdminProductsExtra($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return '';
        }

        $product_id = (int) $params['id_product'];
        $selected_plan_ids = [];

        if ($product_id > 0) {
            // Get saved recurring plan IDs for this product (stored as JSON array)
            $plan_ids_json = Configuration::get('MTBORICA_RECURRING_PLAN_' . $product_id);
            if ($plan_ids_json) {
                $decoded = json_decode($plan_ids_json, true);
                if (is_array($decoded)) {
                    $selected_plan_ids = array_map('intval', $decoded);
                } else {
                    // Backward compatibility: if it's a single integer, convert to array
                    $single_id = (int) $plan_ids_json;
                    if ($single_id > 0) {
                        $selected_plan_ids = [$single_id];
                    }
                }
            }
        }

        // Get all active recurring plans
        require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';
        $recurring_plans = MtboricaRecurring::getAll(true); // true = active only

        // Format plans for display
        $formatted_plans = [];
        foreach ($recurring_plans as $plan) {
            $recurring_obj = new MtboricaRecurring();
            $recurring_obj->hydrate($plan);
            $formatted_plans[] = [
                'id' => $plan['id'],
                'name' => $plan['name'],
                'formatted_duration' => $recurring_obj->getFormattedDuration(),
                'formatted_frequency' => $recurring_obj->getFormattedFrequency(),
            ];
        }

        $this->context->smarty->assign([
            'recurring_plans' => $formatted_plans,
            'selected_plan_ids' => $selected_plan_ids,
        ]);

        return $this->display(__FILE__, 'views/templates/admin/product_recurring.tpl');
    }

    /**
     * Hook: Save product data
     * Saves recurring plan ID when product is updated
     * Only processes if recurring feature is enabled
     *
     * @param array $params Hook parameters
     * @return void
     */
    public function hookActionProductUpdate($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return;
        }

        if (!isset($params['id_product'])) {
            return;
        }

        $product_id = (int) $params['id_product'];

        if ($product_id <= 0) {
            return;
        }

        $config_key = 'MTBORICA_RECURRING_PLAN_' . $product_id;

        // Get plan IDs from POST - handle both array and single value
        $plan_ids = [];
        if (isset($_POST['mtborica_recurring_plan_ids']) && is_array($_POST['mtborica_recurring_plan_ids'])) {
            $plan_ids = $_POST['mtborica_recurring_plan_ids'];
        } elseif (isset($_POST['mtborica_recurring_plan_ids']) && !empty($_POST['mtborica_recurring_plan_ids'])) {
            // Handle single value (backward compatibility)
            $plan_ids = [$_POST['mtborica_recurring_plan_ids']];
        }

        // Handle array of plan IDs
        if (!empty($plan_ids)) {
            // Filter and validate plan IDs
            $valid_plan_ids = [];
            require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';

            foreach ($plan_ids as $plan_id) {
                $plan_id = (int) $plan_id;
                if ($plan_id > 0) {
                    // Validate that plan exists and is active
                    $plan = MtboricaRecurring::getById($plan_id);
                    if ($plan && $plan->status == 1) {
                        $valid_plan_ids[] = $plan_id;
                    }
                }
            }

            if (!empty($valid_plan_ids)) {
                // Store as JSON array
                Configuration::updateValue($config_key, json_encode($valid_plan_ids));
            } else {
                // Remove plan assignment if no valid plans
                Configuration::deleteByName($config_key);
            }
        } else {
            // Remove plan assignment if no plans selected
            Configuration::deleteByName($config_key);
        }
    }

    /**
     * Get recurring plan IDs for a product
     * Returns empty array if recurring feature is disabled or no plans assigned
     *
     * @param int $product_id Product ID
     * @return array Array of recurring plan IDs (empty if none or feature disabled)
     */
    public static function getProductRecurringPlanIds($product_id)
    {
        // If recurring feature is disabled, always return empty array
        if (!self::isRecurringEnabled()) {
            return [];
        }

        $product_id = (int) $product_id;
        if ($product_id <= 0) {
            return [];
        }

        $plan_ids_json = Configuration::get('MTBORICA_RECURRING_PLAN_' . $product_id);
        if (!$plan_ids_json) {
            return [];
        }

        $decoded = json_decode($plan_ids_json, true);
        if (is_array($decoded)) {
            return array_map('intval', $decoded);
        } else {
            // Backward compatibility: if it's a single integer, convert to array
            $single_id = (int) $plan_ids_json;
            if ($single_id > 0) {
                return [$single_id];
            }
        }

        return [];
    }

    /**
     * Get recurring plan ID for a product (backward compatibility)
     * Returns the first plan ID if multiple plans are assigned, or 0 if none
     * Returns 0 if recurring feature is disabled
     *
     * @param int $product_id Product ID
     * @return int Recurring plan ID (0 if none or feature disabled)
     */
    public static function getProductRecurringPlanId($product_id)
    {
        $plan_ids = self::getProductRecurringPlanIds($product_id);
        if (!empty($plan_ids)) {
            return (int) $plan_ids[0];
        }
        return 0;
    }

    /**
     * Check if product has recurring plan
     * Returns false if recurring feature is disabled
     *
     * @param int $product_id Product ID
     * @return bool True if product has recurring plan and feature is enabled
     */
    public static function productHasRecurringPlan($product_id)
    {
        // If recurring feature is disabled, always return false
        if (!self::isRecurringEnabled()) {
            return false;
        }

        $plan_ids = self::getProductRecurringPlanIds($product_id);
        return !empty($plan_ids);
    }

    /**
     * Check if cart contains any recurring products
     * Returns false if recurring feature is disabled
     *
     * @param Cart|null $cart Cart object (if null, uses context cart)
     * @return bool True if cart contains at least one recurring product
     */
    public static function cartHasRecurringProducts($cart = null)
    {
        // If recurring feature is disabled, always return false
        if (!self::isRecurringEnabled()) {
            return false;
        }

        if ($cart === null) {
            $context = Context::getContext();
            $cart = $context->cart;
        }

        if (!$cart || !$cart->id) {
            return false;
        }

        $products = $cart->getProducts();
        if (empty($products)) {
            return false;
        }

        foreach ($products as $product) {
            $product_id = (int) $product['id_product'];
            if (self::productHasRecurringPlan($product_id)) {
                return true;
            }
        }

        return false;
    }

    /**
     * Check if cart contains only recurring products (no standard products)
     * Returns false if recurring feature is disabled
     *
     * @param Cart|null $cart Cart object (if null, uses context cart)
     * @return bool True if cart contains only recurring products (or is empty)
     */
    public static function cartHasOnlyRecurringProducts($cart = null)
    {
        // If recurring feature is disabled, always return false
        if (!self::isRecurringEnabled()) {
            return false;
        }

        if ($cart === null) {
            $context = Context::getContext();
            $cart = $context->cart;
        }

        if (!$cart || !$cart->id) {
            return false; // Empty cart is not considered "only recurring"
        }

        $products = $cart->getProducts();
        if (empty($products)) {
            return false; // Empty cart is not considered "only recurring"
        }

        foreach ($products as $product) {
            $product_id = (int) $product['id_product'];
            if (!self::productHasRecurringPlan($product_id)) {
                return false; // Found a non-recurring product
            }
        }

        return true; // All products are recurring
    }

    /**
     * Get all recurring products from cart with their plan information
     * Returns empty array if recurring feature is disabled or no recurring products found
     *
     * @param Cart|null $cart Cart object (if null, uses context cart)
     * @return array Array of products with recurring plan info:
     *               [
     *                 'id_product' => int,
     *                 'id_product_attribute' => int,
     *                 'quantity' => int,
     *                 'name' => string,
     *                 'recurring_plan_ids' => array,
     *                 'recurring_plan_id' => int (first plan ID for backward compatibility)
     *               ]
     */
    public static function getCartRecurringProducts($cart = null)
    {
        // If recurring feature is disabled, always return empty array
        if (!self::isRecurringEnabled()) {
            return [];
        }

        if ($cart === null) {
            $context = Context::getContext();
            $cart = $context->cart;
        }

        if (!$cart || !$cart->id) {
            return [];
        }

        $products = $cart->getProducts();
        if (empty($products)) {
            return [];
        }

        $recurring_products = [];

        foreach ($products as $product) {
            $product_id = (int) $product['id_product'];
            $plan_ids = self::getProductRecurringPlanIds($product_id);

            if (!empty($plan_ids)) {
                $recurring_products[] = [
                    'id_product' => $product_id,
                    'id_product_attribute' => (int) ($product['id_product_attribute'] ?? 0),
                    'quantity' => (int) ($product['cart_quantity'] ?? $product['quantity'] ?? 0),
                    'name' => isset($product['name']) ? $product['name'] : '',
                    'recurring_plan_ids' => $plan_ids,
                    'recurring_plan_id' => !empty($plan_ids) ? (int) $plan_ids[0] : 0,
                ];
            }
        }

        return $recurring_products;
    }

    /**
     * Check if cart has mixed products (both recurring and standard)
     * Returns false if recurring feature is disabled
     *
     * @param Cart|null $cart Cart object (if null, uses context cart)
     * @return bool True if cart contains both recurring and standard products
     */
    public static function cartHasMixedProducts($cart = null)
    {
        // If recurring feature is disabled, always return false
        if (!self::isRecurringEnabled()) {
            return false;
        }

        if ($cart === null) {
            $context = Context::getContext();
            $cart = $context->cart;
        }

        if (!$cart || !$cart->id) {
            return false;
        }

        $products = $cart->getProducts();
        if (empty($products)) {
            return false;
        }

        $has_recurring = false;
        $has_standard = false;

        foreach ($products as $product) {
            $product_id = (int) $product['id_product'];
            $product_attribute_id = (int) ($product['id_product_attribute'] ?? 0);

            // Check if this specific product in cart has a selected recurring plan
            // A product is only considered recurring if it has a plan selected in the cart
            $product_plan_id = self::getCartProductRecurringPlan($cart->id, $product_id, $product_attribute_id);

            if ($product_plan_id > 0) {
                // Product has a selected recurring plan in cart
                $has_recurring = true;
            } else {
                // Product doesn't have a selected plan (even if it has recurring capability)
                $has_standard = true;
            }

            // Early exit if both types found
            if ($has_recurring && $has_standard) {
                return true;
            }
        }

        return false; // Not mixed (either all recurring, all standard, or empty)
    }

    /**
     * Save recurring plan ID for a product in cart to database
     * Replaces cookie-based storage with database storage
     *
     * @param int $cart_id Cart ID
     * @param int $product_id Product ID
     * @param int $product_attribute_id Product attribute ID (default: 0)
     * @param int $recurring_plan_id Recurring plan ID (0 = no plan, standard product)
     * @return bool True on success, false on failure
     */
    public static function saveCartProductRecurringPlan($cart_id, $product_id, $product_attribute_id = 0, $recurring_plan_id = 0)
    {
        if (!self::isRecurringEnabled()) {
            return false;
        }

        $cart_id = (int) $cart_id;
        $product_id = (int) $product_id;
        $product_attribute_id = (int) $product_attribute_id;
        $recurring_plan_id = (int) $recurring_plan_id;

        if ($cart_id <= 0 || $product_id <= 0) {
            return false;
        }

        // Attempt to resolve product attribute when not provided (e.g., combinations)
        if ($product_attribute_id <= 0) {
            $cart = new Cart($cart_id);
            if (Validate::isLoadedObject($cart)) {
                $cart_products = $cart->getProducts();
                $matching_rows = [];
                foreach ($cart_products as $product_row) {
                    if ((int) ($product_row['id_product'] ?? 0) === $product_id) {
                        $matching_rows[] = $product_row;
                    }
                }

                if (count($matching_rows) === 1) {
                    $product_attribute_id = (int) ($matching_rows[0]['id_product_attribute'] ?? 0);
                } else {
                    foreach ($matching_rows as $product_row) {
                        $candidate_attr = (int) ($product_row['id_product_attribute'] ?? 0);
                        if ($candidate_attr > 0) {
                            $product_attribute_id = $candidate_attr;
                            break;
                        }
                    }
                }
            }
        }

        $db = Db::getInstance();
        $table = _DB_PREFIX_ . 'mtborica_cart_products';

        // Check if record exists
        $sql = 'SELECT `id` FROM `' . $table . '` 
                WHERE `id_cart` = ' . (int) $cart_id . '
                AND `id_product` = ' . (int) $product_id . '
                AND `id_product_attribute` = ' . (int) $product_attribute_id;
        $existing = $db->getValue($sql);

        if ($recurring_plan_id <= 0) {
            // If plan ID is 0, delete the record (standard product)
            if ($existing) {
                return $db->delete('mtborica_cart_products', 'id = ' . (int) $existing);
            }
            return true; // Already doesn't exist
        }

        if ($existing) {
            // Update existing record
            $sql = 'UPDATE `' . $table . '` 
                    SET `id_recurring_plan` = ' . (int) $recurring_plan_id . '
                    WHERE `id` = ' . (int) $existing;
            return (bool) $db->execute($sql);
        } else {
            // Insert new record
            return $db->insert('mtborica_cart_products', [
                'id_cart' => $cart_id,
                'id_product' => $product_id,
                'id_product_attribute' => $product_attribute_id,
                'id_recurring_plan' => $recurring_plan_id,
            ]);
        }
    }

    /**
     * Get recurring plan ID for a product in cart from database
     *
     * @param int $cart_id Cart ID
     * @param int $product_id Product ID
     * @param int $product_attribute_id Product attribute ID (default: 0)
     * @return int Recurring plan ID (0 if not found or no plan)
     */
    public static function getCartProductRecurringPlan($cart_id, $product_id, $product_attribute_id = 0)
    {
        if (!self::isRecurringEnabled()) {
            return 0;
        }

        $cart_id = (int) $cart_id;
        $product_id = (int) $product_id;
        $product_attribute_id = (int) $product_attribute_id;

        if ($cart_id <= 0 || $product_id <= 0) {
            return 0;
        }

        $db = Db::getInstance();
        $sql = 'SELECT `id_recurring_plan` FROM `' . _DB_PREFIX_ . 'mtborica_cart_products` 
                WHERE `id_cart` = ' . (int) $cart_id . '
                AND `id_product` = ' . (int) $product_id . '
                AND `id_product_attribute` = ' . (int) $product_attribute_id;

        $plan_id = (int) $db->getValue($sql);

        if ($plan_id <= 0 && $product_attribute_id > 0) {
            $fallback_sql = 'SELECT `id_recurring_plan` FROM `' . _DB_PREFIX_ . 'mtborica_cart_products` 
                WHERE `id_cart` = ' . (int) $cart_id . '
                AND `id_product` = ' . (int) $product_id . '
                AND `id_product_attribute` = 0';
            $plan_id = (int) $db->getValue($fallback_sql);
        }

        return $plan_id > 0 ? $plan_id : 0;
    }

    /**
     * Get all recurring plan IDs for products in a cart
     *
     * @param int $cart_id Cart ID
     * @return array Array of plan IDs (empty if none found)
     */
    public static function getCartRecurringPlanIds($cart_id)
    {
        if (!self::isRecurringEnabled()) {
            return [];
        }

        $cart_id = (int) $cart_id;
        if ($cart_id <= 0) {
            return [];
        }

        $db = Db::getInstance();
        $sql = 'SELECT DISTINCT `id_recurring_plan` FROM `' . _DB_PREFIX_ . 'mtborica_cart_products` 
                WHERE `id_cart` = ' . (int) $cart_id . '
                AND `id_recurring_plan` > 0';

        $results = $db->executeS($sql);
        if (empty($results)) {
            return [];
        }

        $plan_ids = [];
        foreach ($results as $row) {
            $plan_id = (int) $row['id_recurring_plan'];
            if ($plan_id > 0) {
                $plan_ids[] = $plan_id;
            }
        }

        return array_unique($plan_ids);
    }

    /**
     * Check if cart contains only recurring products with the same plan
     * Returns the plan ID if all products use the same recurring plan, 0 otherwise
     *
     * @param Cart|null $cart Cart object (default: current context cart)
     * @return int Recurring plan ID if all products use the same plan, 0 otherwise
     */
    public static function cartHasSingleRecurringPlan($cart = null)
    {
        // If recurring feature is disabled, always return 0
        if (!self::isRecurringEnabled()) {
            return 0;
        }

        if ($cart === null) {
            $context = Context::getContext();
            $cart = $context->cart;
        }

        if (!$cart || !$cart->id) {
            return 0;
        }

        // Get all recurring plan IDs from cart
        $plan_ids = self::getCartRecurringPlanIds($cart->id);

        // If no recurring plans found, return 0
        if (empty($plan_ids)) {
            return 0;
        }

        // If more than one different plan found, return 0
        if (count($plan_ids) > 1) {
            return 0;
        }

        // Check if all products in cart have recurring plans
        $products = $cart->getProducts();
        if (empty($products)) {
            return 0;
        }

        $single_plan_id = (int) $plan_ids[0];
        $all_products_have_plan = true;

        foreach ($products as $product) {
            $product_id = (int) $product['id_product'];
            $product_attribute_id = (int) ($product['id_product_attribute'] ?? 0);

            // Check if product has recurring plan
            if (!self::productHasRecurringPlan($product_id)) {
                // Product doesn't have recurring plan - not all products are recurring
                $all_products_have_plan = false;
                break;
            }

            // Check if this specific product in cart has the same plan
            $product_plan_id = self::getCartProductRecurringPlan($cart->id, $product_id, $product_attribute_id);
            if ($product_plan_id != $single_plan_id) {
                // Product has different plan or no plan
                $all_products_have_plan = false;
                break;
            }
        }

        return $all_products_have_plan ? $single_plan_id : 0;
    }

    /**
     * Delete recurring plan records for a cart (when cart is deleted or cleared)
     *
     * @param int $cart_id Cart ID
     * @return bool True on success, false on failure
     */
    public static function deleteCartRecurringPlans($cart_id)
    {
        $cart_id = (int) $cart_id;
        if ($cart_id <= 0) {
            return false;
        }

        $db = Db::getInstance();
        return $db->delete('mtborica_cart_products', 'id_cart = ' . (int) $cart_id);
    }

    /**
     * Clean up orphaned recurring plan records from database
     * Removes records for:
     * - Carts that no longer exist
     * - Carts that are empty (no products)
     * - Carts older than specified days (default: 30 days)
     *
     * This method should be called periodically to prevent database bloat.
     * Can be called via cron job or during cart operations.
     *
     * @param int $days_old Delete records for carts older than this many days (default: 30)
     * @return array Statistics: ['deleted_nonexistent' => int, 'deleted_empty' => int, 'deleted_old' => int, 'total_deleted' => int]
     */
    public static function cleanupOrphanedCartPlanRecords($days_old = 1)
    {
        if (!self::isRecurringEnabled()) {
            return [
                'deleted_nonexistent' => 0,
                'deleted_empty' => 0,
                'deleted_old' => 0,
                'total_deleted' => 0,
            ];
        }

        $db = Db::getInstance();
        $table = _DB_PREFIX_ . 'mtborica_cart_products';
        $cart_table = _DB_PREFIX_ . 'cart';
        $stats = [
            'deleted_nonexistent' => 0,
            'deleted_empty' => 0,
            'deleted_old' => 0,
            'total_deleted' => 0,
        ];

        // Get all unique cart IDs from our table
        $sql = 'SELECT DISTINCT `id_cart` FROM `' . $table . '`';
        $cart_ids = $db->executeS($sql);

        if (empty($cart_ids)) {
            return $stats;
        }

        $cutoff_date = date('Y-m-d H:i:s', strtotime('-' . (int) $days_old . ' days'));

        // Remove plan records whose own update timestamp is stale
        $deleted_stale_records = $db->delete('mtborica_cart_products', "`date_upd` < '" . pSQL($cutoff_date) . "'");
        if ($deleted_stale_records) {
            $stats['deleted_old'] += $deleted_stale_records;
        }

        foreach ($cart_ids as $row) {
            $cart_id = (int) $row['id_cart'];
            if ($cart_id <= 0) {
                continue;
            }

            // Check if cart exists
            $cart_exists = $db->getValue(
                'SELECT COUNT(*) FROM `' . $cart_table . '` WHERE `id_cart` = ' . (int) $cart_id
            );

            if (!$cart_exists) {
                // Cart doesn't exist - delete all records for this cart
                $deleted = $db->delete('mtborica_cart_products', 'id_cart = ' . (int) $cart_id);
                if ($deleted) {
                    $stats['deleted_nonexistent'] += $deleted;
                }
                continue;
            }

            // Check if cart is empty
            $cart = new Cart($cart_id);
            if (Validate::isLoadedObject($cart)) {
                $products = $cart->getProducts();
                if (empty($products)) {
                    // Cart is empty - delete all records for this cart
                    $deleted = $db->delete('mtborica_cart_products', 'id_cart = ' . (int) $cart_id);
                    if ($deleted) {
                        $stats['deleted_empty'] += $deleted;
                    }
                    continue;
                }

                // Check if cart is old (based on date_upd)
                $cart_date_upd = $db->getValue(
                    'SELECT `date_upd` FROM `' . $cart_table . '` WHERE `id_cart` = ' . (int) $cart_id
                );

                if ($cart_date_upd && $cart_date_upd < $cutoff_date) {
                    // Cart is old - delete all records for this cart
                    $deleted = $db->delete('mtborica_cart_products', 'id_cart = ' . (int) $cart_id);
                    if ($deleted) {
                        $stats['deleted_old'] += $deleted;
                    }
                }
            }
        }

        $stats['total_deleted'] = $stats['deleted_nonexistent'] + $stats['deleted_empty'] + $stats['deleted_old'];

        return $stats;
    }

    /**
     * Hook: Called when cart is saved/updated
     * Saves recurring plan ID for products added to cart
     * Only processes if recurring feature is enabled
     *
     * @param array $params Hook parameters (contains 'cart')
     * @return void
     */
    public function hookActionCartSave($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return;
        }

        if (!isset($params['cart']) || !$params['cart']->id) {
            return;
        }

        $cart = $params['cart'];
        $cart_id = (int) $cart->id;

        // Get plan ID from POST (sent by JavaScript after validation)
        // Only process if mtborica_recurring_plan_id is explicitly sent in POST
        // This prevents deletion of existing plan when quantity is changed in cart
        if (!isset($_POST['mtborica_recurring_plan_id'])) {
            // Plan ID not sent in POST - this is likely a quantity change or product removal
            // Clean up orphaned records: remove plan records for products no longer in cart
            self::cleanupOrphanedCartPlans($cart_id);

            return;
        }

        $product_id = (int) Tools::getValue('id_product', 0);
        $product_attribute_id = (int) Tools::getValue('id_product_attribute', 0);
        $recurring_plan_id = (int) Tools::getValue('mtborica_recurring_plan_id', 0);

        // Only save if product ID is provided and plan ID was explicitly sent
        if ($product_id > 0) {
            self::saveCartProductRecurringPlan($cart_id, $product_id, $product_attribute_id, $recurring_plan_id);
        }
    }

    /**
     * Clean up orphaned recurring plan records for products no longer in cart
     * Called when cart is updated but plan ID is not sent in POST (quantity change or product removal)
     *
     * @param int $cart_id Cart ID
     * @return void
     */
    public static function cleanupOrphanedCartPlans($cart_id)
    {
        if ($cart_id <= 0) {
            return;
        }

        // Get all products currently in cart
        $cart = new Cart($cart_id);
        if (!Validate::isLoadedObject($cart)) {
            return;
        }

        $cart_products = $cart->getProducts();
        $cart_product_keys = [];

        // Create keys for products in cart: "product_id-attribute_id"
        foreach ($cart_products as $product) {
            $product_id = (int) $product['id_product'];
            $attribute_id = (int) $product['id_product_attribute'];
            $cart_product_keys[] = $product_id . '-' . $attribute_id;
        }

        // Get all plan records for this cart from database
        $db = Db::getInstance();
        $table = _DB_PREFIX_ . 'mtborica_cart_products';
        $sql = 'SELECT `id`, `id_product`, `id_product_attribute` FROM `' . $table . '` 
                WHERE `id_cart` = ' . (int) $cart_id;
        $plan_records = $db->executeS($sql);

        if (empty($plan_records)) {
            return;
        }

        // Check each plan record - if product is not in cart, delete the record
        foreach ($plan_records as $record) {
            $product_id = (int) $record['id_product'];
            $attribute_id = (int) $record['id_product_attribute'];
            $record_key = $product_id . '-' . $attribute_id;

            // If product is not in cart, delete the plan record
            if (!in_array($record_key, $cart_product_keys)) {
                $db->delete('mtborica_cart_products', 'id = ' . (int) $record['id']);
            }
        }
    }

    /**
     * Hook: Display content before product buttons (Add to cart)
     * Shows recurring plan selection dropdown if product has recurring plans
     * Only shows if recurring feature is enabled
     *
     * @param array $params Hook parameters (contains 'product')
     * @return string HTML content (empty if recurring is disabled or no plans)
     */
    public function hookDisplayProductButtons($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return '';
        }

        if (!isset($params['product'])) {
            return '';
        }

        // Handle both object and array formats
        $product = $params['product'];
        if (is_object($product)) {
            $product_id = (int) $product->id;
        } elseif (is_array($product) && isset($product['id'])) {
            $product_id = (int) $product['id'];
        } else {
            return '';
        }

        if ($product_id <= 0) {
            return '';
        }

        // Generate validation URL (always needed, even for products without plans)
        $validate_url = $this->context->link->getModuleLink(
            $this->name,
            'validateAddToCart',
            []
        );

        // Generate save plan URL (always needed, even for products without plans)
        $save_plan_url = $this->context->link->getModuleLink(
            $this->name,
            'saveCartPlan',
            []
        );

        // Get recurring plan IDs for this product
        $plan_ids = self::getProductRecurringPlanIds($product_id);
        $recurring_plans = [];

        // Only get plan details if product has recurring plans
        if (!empty($plan_ids)) {
            // Get plan details
            require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';

            foreach ($plan_ids as $plan_id) {
                $plan = MtboricaRecurring::getById($plan_id);
                if ($plan && $plan->status == 1) {
                    $recurring_obj = new MtboricaRecurring();
                    $recurring_obj->hydrate((array) $plan);
                    $recurring_plans[] = [
                        'id' => $plan->id,
                        'name' => $plan->name,
                        'formatted_duration' => $recurring_obj->getFormattedDuration(),
                        'formatted_frequency' => $recurring_obj->getFormattedFrequency(),
                    ];
                }
            }
        }

        // Always show validation code, even for products without recurring plans
        // This ensures validation is called for all products
        $this->context->smarty->assign([
            'product_id' => $product_id,
            'recurring_plans' => $recurring_plans,
            'validate_url' => $validate_url,
            'save_plan_url' => $save_plan_url,
            'has_recurring_plans' => !empty($recurring_plans),
        ]);

        return $this->display(__FILE__, 'views/templates/front/product_recurring_selector.tpl');
    }

    /**
     * Hook: Display additional information for product in shopping cart (alternative hook)
     * Shows recurring plan name if product has a recurring plan selected
     * This is a fallback hook in case displayShoppingCartProduct is not called
     *
     * @param array $params Hook parameters
     * @return string HTML content
     */
    public function hookDisplayProductPriceBlock($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return '';
        }

        // Only show on cart page and for unit_price type
        if (!isset($params['type']) || $params['type'] !== 'unit_price') {
            return '';
        }

        // Get product from params - can be array or object
        if (!isset($params['product'])) {
            return '';
        }

        $product = $params['product'];

        // Handle both array and object
        if (is_array($product)) {
            $product_id = (int) ($product['id_product'] ?? 0);
            $product_attribute_id = (int) ($product['id_product_attribute'] ?? 0);
        } elseif (is_object($product)) {
            $product_id = (int) ($product->id_product ?? 0);
            $product_attribute_id = (int) ($product->id_product_attribute ?? 0);
        } else {
            return '';
        }

        if ($product_id <= 0) {
            return '';
        }

        // Check if product has recurring plans
        if (!self::productHasRecurringPlan($product_id)) {
            return '';
        }

        // Get cart ID from context - try multiple methods for AJAX compatibility
        $cart_id = 0;

        // Method 1: Try from context cart object (normal page load)
        if (isset($this->context->cart) && is_object($this->context->cart) && isset($this->context->cart->id)) {
            $cart_id = (int) $this->context->cart->id;
        }

        // Method 2: Try from cookie if context cart is not available (AJAX case)
        if ($cart_id <= 0 && isset($this->context->cookie) && isset($this->context->cookie->id_cart)) {
            $cookie_cart_id = (int) $this->context->cookie->id_cart;
            if ($cookie_cart_id > 0) {
                // Verify cart exists and is valid
                $cart = new Cart($cookie_cart_id);
                if (Validate::isLoadedObject($cart)) {
                    $cart_id = (int) $cart->id;
                }
            }
        }

        if ($cart_id <= 0) {
            return '';
        }

        // Get selected plan ID from database
        $selected_plan_id = self::getCartProductRecurringPlan($cart_id, $product_id, $product_attribute_id);

        // If no plan selected, don't show anything
        if ($selected_plan_id <= 0) {
            return '';
        }

        // Load plan information
        require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaRecurring.php';
        $plan = MtboricaRecurring::getById($selected_plan_id);

        if (!$plan || !$plan->id) {
            return '';
        }

        // Assign variables to template
        $this->context->smarty->assign([
            'recurring_plan_name' => $plan->name,
            'recurring_plan_id' => $selected_plan_id,
        ]);

        return $this->display(__FILE__, 'views/templates/front/cart_recurring_plan.tpl');
    }

    /**
     * Hook: displayOrderDetail
     * Displays "Close recurring plan" button on order detail page for recurring payments
     *
     * @param array $params Hook parameters (contains order object)
     * @return string HTML content (empty if conditions not met)
     */
    public function hookDisplayOrderDetail($params)
    {
        // Check if recurring feature is enabled
        if (!self::isRecurringEnabled()) {
            return '';
        }

        // Check if order exists and uses our payment method
        if (empty($params['order']) || !is_object($params['order'])) {
            return '';
        }

        /** @var Order $order */
        $order = $params['order'];
        if (!Validate::isLoadedObject($order) || $order->module !== $this->name) {
            return '';
        }

        // Check if customer is logged in and owns this order
        if (
            !$this->context->customer->isLogged() ||
            (int) $this->context->customer->id !== (int) $order->id_customer
        ) {
            return '';
        }

        // Get payment data from database
        $paymentData = $this->getOrderPaymentData($order->id);
        if (!$paymentData || empty($paymentData['recur_id']) || (int) $paymentData['recur_id'] <= 0) {
            return ''; // Not a recurring payment
        }

        // Check if order status is recurring (not cancelled)
        $recurring_status_id = (int) Configuration::get('mtborica_order_status_recurring');

        $current_status_id = (int) $order->getCurrentState();

        // Only show button if order is in recurring status (not cancelled)
        if ($current_status_id !== $recurring_status_id) {
            return '';
        }

        // Check if plan is already cancelled (check request_cancel status)
        if (!empty($paymentData['request_cancel']) && $paymentData['request_cancel'] === '00') {
            return ''; // Already cancelled
        }

        // Load recurring plan information
        $recurring_plan = MtboricaRecurring::getById($paymentData['recur_id']);
        if (!$recurring_plan || !$recurring_plan->id) {
            return '';
        }

        // Prepare BORICA cancellation form data
        $currency = new Currency((int) $order->id_currency);
        $borica_currency = $currency->iso_code;

        $borica_terminal = '';
        $borica_merchant = '';
        if ('BGN' === $borica_currency) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_bgn');
            $borica_merchant = (string) Configuration::get('mtborica_mid_bgn');
        }
        if ('EUR' === $borica_currency) {
            $borica_terminal = (string) Configuration::get('mtborica_tid_eur');
            $borica_merchant = (string) Configuration::get('mtborica_mid_eur');
        }

        if (empty($borica_terminal) || empty($borica_merchant)) {
            return ''; // Cannot proceed without terminal/merchant
        }

        // Prepare order number
        $order_id = $order->id;
        if (strlen($order_id) >= 6) {
            $borica_order = substr($order_id, -6);
        } else {
            $borica_order = str_pad($order_id, 6, '0', STR_PAD_LEFT);
        }

        // Get BORICA URL
        $borica_testmode = (int) Configuration::get('mtborica_testmode');
        $borica_url = $borica_testmode === 1
            ? self::BORICA_TEST_URL
            : self::BORICA_PRODUCTION_URL;

        // Prepare recurring plan cancellation parameters
        $borica_trtype = self::BORICA_TRTYPE_RECURRING_DROP_STATUS; // 179
        $borica_timestamp = gmdate('YmdHis');
        $borica_nonce = !empty($paymentData['nonce']) ? $paymentData['nonce'] : '';

        // Get data from payment record
        $borica_amount = !empty($paymentData['amount']) ? number_format((float) $paymentData['amount'], 2, '.', '') : '';
        $borica_int_ref = !empty($paymentData['int_ref']) ? $paymentData['int_ref'] : '';
        $borica_rrn = !empty($paymentData['rrn']) ? $paymentData['rrn'] : '';
        $borica_merch_rn_id = !empty($paymentData['merch_rn_id']) ? $paymentData['merch_rn_id'] : '';

        if (empty($borica_int_ref) || empty($borica_rrn)) {
            return ''; // Missing required payment data
        }

        // Prepare other BORICA fields
        $base_url = rtrim(Tools::getShopDomainSsl(true, true), '/');
        $borica_desc = $this->l('Partial payment on order from') . ' ' . $order_id;
        $borica_merch_name = (string) Configuration::get('mtborica_mname');
        $borica_merchant_url = $base_url;
        $borica_email = (string) Configuration::get('mtborica_email');
        $borica_country = self::BORICA_COUNTRY;
        $borica_lang = (string) Configuration::get('mtborica_payment_lang');
        $borica_addendum = self::BORICA_ADDENDUM;
        $borica_ad_cust_bor_order_id = $borica_order . $order_id;

        // Sign the request using signDropPayment
        require_once _PS_MODULE_DIR_ . $this->name . '/classes/MtboricaBoricaHelper.php';
        $sign_result = MtboricaBoricaHelper::signDropPayment(
            $borica_terminal,
            $borica_trtype,
            $borica_amount,
            $borica_currency,
            $borica_order,
            $borica_timestamp,
            $borica_nonce
        );

        if (!empty($sign_result['boricaError'])) {
            return ''; // Cannot sign request
        }

        $borica_p_sign = $sign_result['pSign'];

        // Prepare form data array
        $borica_form_data = [
            'TERMINAL' => $borica_terminal,
            'TRTYPE' => (string) $borica_trtype,
            'AMOUNT' => $borica_amount,
            'CURRENCY' => $borica_currency,
            'DESC' => $borica_desc,
            'MERCHANT' => $borica_merchant,
            'MERCH_NAME' => $borica_merch_name,
            'MERCH_URL' => $borica_merchant_url,
            'EMAIL' => $borica_email,
            'COUNTRY' => $borica_country,
            'LANG' => $borica_lang,
            'ADDENDUM' => $borica_addendum,
            'AD.CUST_BOR_ORDER_ID' => $borica_ad_cust_bor_order_id,
            'TIMESTAMP' => $borica_timestamp,
            'ORDER' => $borica_order,
            'INT_REF' => $borica_int_ref,
            'RRN' => $borica_rrn,
            'MERCH_RN_ID' => $borica_merch_rn_id,
            'NONCE' => $borica_nonce,
            'P_SIGN' => $borica_p_sign,
        ];

        if (1 === (int) Configuration::get('mtborica_debug')) {
            MtboricaLogger::logCancelRecurringPaymentSent($borica_form_data);
        }

        // Prepare template variables
        $this->context->smarty->assign([
            'order_id' => (int) $order->id,
            'recurring_plan_name' => $recurring_plan->name,
            'borica_url' => $borica_url,
            'borica_form_data' => $borica_form_data,
        ]);

        return $this->display(__FILE__, 'views/templates/front/order_detail_recurring_cancel.tpl');
    }

    private function installOrderStates()
    {
        return $this->createOrderState(
            'mtborica_order_status_recurring',
            'Recurring payment',
            '#3d70b2',
            true,
            true,
            false
        ) &&
            $this->createOrderState(
                'mtborica_order_status_recurring_cancelled',
                'Cancelled Recurring payments',
                '#d94452',
                false,
                false,
                true
            );
    }

    private function uninstallOrderStates()
    {
        $keys = [
            'mtborica_order_status_recurring',
            'mtborica_order_status_recurring_cancelled',
        ];

        foreach ($keys as $configKey) {
            $id_state = (int) Configuration::get($configKey);
            if ($id_state > 0) {
                $orderState = new OrderState($id_state);
                if (Validate::isLoadedObject($orderState)) {
                    $orderState->delete();
                }
            }
        }

        return true;
    }

    private function createOrderState($configKey, $name, $color, $logable, $invoice, $hidden)
    {
        $id_state = (int) Configuration::get($configKey);
        if ($id_state > 0 && OrderState::existsInDatabase($id_state, 'order_state')) {
            return true;
        }

        $existingId = (int) Db::getInstance()->getValue(
            (new DbQuery())
                ->select('id_order_state')
                ->from('order_state')
                ->where('module_name = \'' . pSQL($this->name) . '\'')
                ->where('hidden = ' . (int) $hidden)
                ->orderBy('id_order_state DESC')
        );

        if ($existingId > 0 && OrderState::existsInDatabase($existingId, 'order_state')) {
            $orderState = new OrderState($existingId);
            if (Validate::isLoadedObject($orderState)) {
                $orderState->color = $color;
                $orderState->logable = $logable;
                $orderState->invoice = $invoice;
                $orderState->send_email = false;
                $orderState->hidden = $hidden;
                $orderState->unremovable = false;
                $orderState->active = true;
                $orderState->module_name = $this->name;

                $languages = Language::getLanguages(false);
                foreach ($languages as $language) {
                    $idLang = (int) $language['id_lang'];
                    $orderState->name[$idLang] = $this->l($name);
                }

                $orderState->update();
            }

            Configuration::updateValue($configKey, $existingId);

            return true;
        }

        /** @var OrderState $orderState */
        $orderState = new OrderState();
        $orderState->color = $color;
        $orderState->logable = $logable;
        $orderState->invoice = $invoice;
        $orderState->send_email = false;
        $orderState->hidden = $hidden;
        $orderState->unremovable = false;
        $orderState->active = true;
        $orderState->module_name = $this->name;

        /** @var array<int, array<string, mixed>> $languages */
        $languages = Language::getLanguages(false);
        foreach ($languages as $language) {
            $idLang = (int) $language['id_lang'];
            $orderState->name[$idLang] = $this->l($name);
        }

        if (!$orderState->add()) {
            return false;
        }

        Configuration::updateValue($configKey, (int) $orderState->id);

        return true;
    }
}
